<?php
/**
 * توابع عمومی ربات
 * نسخه ارتقاءیافته (V2)
 * - پروتکل واحد: V2RAY
 * - منوی جداگانه ادمین/کاربر
 * - State Machine یکپارچه با ذخیره timestamp
 */

mb_internal_encoding('UTF-8');

$config = require __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

define('BOT_TOKEN', $config['BOT_TOKEN']);
define('API', 'https://api.telegram.org/bot' . BOT_TOKEN . '/');
define('ADMIN_ID', (int)$config['ADMIN_ID']);
define('MAIN_PROTOCOL', 'V2RAY');

// ================== تنظیمات زیرمجموعه‌گیری ==================
define('REFERRAL_COMMISSION_PERCENT', 5); // درصد پورسانت از خرید زیرمجموعه
define('REFERRAL_MEMBERSHIP_GIFT_TOMAN', (int)($config['REFERRAL_MEMBERSHIP_GIFT_TOMAN'] ?? 10000)); // هدیه عضویت (تومان)


/**
 * لاگ خطاها در فایل
 */
function log_error($message, array $context = []) {
    $dir = __DIR__ . '/logs';
    if (!is_dir($dir)) {
        @mkdir($dir, 0755, true);
    }
    $line = '[' . date('Y-m-d H:i:s') . '] ' . $message;
    if (!empty($context)) {
        $line .= ' | ' . json_encode($context, JSON_UNESCAPED_UNICODE);
    }
    $line .= "\n";
    @file_put_contents($dir . '/error.log', $line, FILE_APPEND);
}

function tg($method, $params = []) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => API . $method,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_POSTFIELDS => json_encode($params, JSON_UNESCAPED_UNICODE),
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT => 40,
    ]);
    $res = curl_exec($ch);
    if ($res === false) {
        log_error('CURL ERROR', ['err' => curl_error($ch), 'method' => $method]);
        curl_close($ch);
        return null;
    }
    curl_close($ch);

    $j = json_decode($res, true);
    if (!$j || !isset($j['ok'])) {
        log_error('API BAD RESPONSE', ['method' => $method, 'raw' => $res]);
        return null;
    }
    if (!$j['ok']) {
        // بعضی خطاهای 400 در تلگرام «بی‌خطر» هستند و فقط باعث نویز در لاگ می‌شوند.
        // این‌ها را لاگ نمی‌کنیم:
        // - message is not modified
        // - there is no text in the message to edit (وقتی پیام مدیا است و باید caption ادیت شود)
        // - object expected as reply markup (وقتی reply_markup=null ارسال شود)
        $desc = (string)($j['description'] ?? '');
        $suppress = false;
        if (isset($j['error_code']) && (int)$j['error_code'] === 400) {
            if (strpos($desc, 'message is not modified') !== false) $suppress = true;
            if ($method === 'editMessageText' && strpos($desc, 'there is no text in the message to edit') !== false) $suppress = true;
            if ($method === 'editMessageReplyMarkup' && strpos($desc, 'object expected as reply markup') !== false) $suppress = true;
        }
        if (!$suppress) {
            log_error('API ERROR', ['method' => $method, 'resp' => $j]);
        }
    }
    return $j;
}

function sendMessage($chat_id, $text, $reply_markup = null, $parse_mode = 'HTML', $reply_to = null) {
    $params = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => $parse_mode,
        'disable_web_page_preview' => true,
    ];
    if ($reply_markup) $params['reply_markup'] = $reply_markup;
    if ($reply_to) $params['reply_to_message_id'] = $reply_to;
    return tg('sendMessage', $params);
}


function sendPhoto($chat_id, $photo, $caption = '', $reply_markup = null, $parse_mode = 'HTML') {
    $params = [
        'chat_id' => $chat_id,
        'photo' => $photo,
        'caption' => $caption,
        'parse_mode' => $parse_mode,
    ];
    if ($reply_markup) $params['reply_markup'] = $reply_markup;
    return tg('sendPhoto', $params);
}

function editMessageText($chat_id, $message_id, $text, $reply_markup = null, $parse_mode = 'HTML') {
    $params = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => $text,
        'parse_mode' => $parse_mode,
        'disable_web_page_preview' => true,
    ];
    if ($reply_markup) $params['reply_markup'] = $reply_markup;
    // اگر پیام مدیا (عکس/ویدیو/...) باشد، تلگرام اجازه editMessageText نمی‌دهد.
    // در این حالت به صورت خودکار editMessageCaption را امتحان می‌کنیم.
    $res = tg('editMessageText', $params);
    if (is_array($res) && isset($res['ok']) && $res['ok'] === false) {
        $desc = (string)($res['description'] ?? '');
        if (strpos($desc, 'there is no text in the message to edit') !== false) {
            return editMessageCaption($chat_id, $message_id, $text, $reply_markup, $parse_mode);
        }
    }
    return $res;
}


/**
 * ویرایش کپشن پیام‌های مدیا (عکس/ویدیو/فایل و...)
 * نکته: برای پیام عکس، editMessageText کار نمی‌کند و باید از editMessageCaption استفاده شود.
 */
function editMessageCaption($chat_id, $message_id, $caption, $reply_markup = null, $parse_mode = 'HTML') {
    $params = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'caption' => $caption,
        'parse_mode' => $parse_mode,
    ];
    if ($reply_markup) $params['reply_markup'] = $reply_markup;
    return tg('editMessageCaption', $params);
}


function editMessageReplyMarkup($chat_id, $message_id, $reply_markup = null) {
    // برای حذف دکمه‌های اینلاین، تلگرام انتظار «object» دارد نه null.
    // بنابراین اگر null بود، یک اینلاین‌کیبورد خالی می‌فرستیم.
    if ($reply_markup === null) {
        $reply_markup = ['inline_keyboard' => []];
    }
    $params = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'reply_markup' => $reply_markup,
    ];
    return tg('editMessageReplyMarkup', $params);
}

/**
 * اطمینان از وجود تگ در انتهای لینک (به صورت #fragment)
 * - اگر لینک آخرش #دژنت باشد دست نمی‌زند
 * - اگر لینک fragment دیگری داشته باشد، همان را با #دژنت جایگزین می‌کند (تا دو تا # ایجاد نشود)
 */
function ensureLinkHasDNetTag($link, $tag = '#دژنت') {
    $link = trim((string)$link);
    if ($link === '') return '';
    $tag = (string)$tag;

    // اگر از قبل در انتها باشد
    if (mb_substr($link, -mb_strlen($tag)) === $tag) {
        return $link;
    }

    // اگر قبلاً fragment داشته باشد، جایگزین کنیم
    $pos = mb_strpos($link, '#');
    if ($pos !== false) {
        $base = mb_substr($link, 0, $pos);
        return $base . $tag;
    }

    // در غیر این صورت اضافه کن
    return $link . $tag;
}

// ================== مانیتورینگ مصرف/انقضا (Usage Monitor) ==================

/**
 * ساخت جدول کش مصرف اگر وجود ندارد (بدون نیاز به اجرای migration دستی)
 */
function ensureConfigUsageCacheTable() {
    try {
        $pdo = db();
        $pdo->exec(
            "CREATE TABLE IF NOT EXISTS config_usage_cache (\n"
            . "  config_id INT PRIMARY KEY,\n"
            . "  used_bytes BIGINT NOT NULL DEFAULT 0,\n"
            . "  total_bytes BIGINT NOT NULL DEFAULT 0,\n"
            . "  expire_ts BIGINT NULL,\n"
            . "  percent_used INT NOT NULL DEFAULT 0,\n"
            . "  days_left INT NULL,\n"
            . "  last_checked BIGINT NOT NULL DEFAULT 0,\n"
            . "  delivered_at BIGINT NULL,\n"
            . "  no_use_notice_sent TINYINT(1) NOT NULL DEFAULT 0,\n"
            . "  warn80_sent TINYINT(1) NOT NULL DEFAULT 0,\n"
            . "  warn10_sent TINYINT(1) NOT NULL DEFAULT 0,\n"
            . "  expired_sent TINYINT(1) NOT NULL DEFAULT 0,\n"
            . "  last_warned_at BIGINT NULL,\n"
            . "  meta JSON NULL,\n"
            . "  CONSTRAINT fk_cfg_usage_cfg FOREIGN KEY (config_id) REFERENCES configs(id) ON DELETE CASCADE\n"
            . ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4"
        );

        // اگر جدول از نسخه قدیمی‌تر ساخته شده باشد، ستون‌های جدید را اضافه کن
        ensureConfigUsageCacheColumns();
    } catch (Throwable $e) {
        // اگر دسترسی ALTER/CREATE ندارید، صرفاً لاگ می‌کنیم
        log_error('ensureConfigUsageCacheTable failed', ['msg' => $e->getMessage()]);
    }
}

/**
 * افزودن ستون‌ها/ایندکس‌های جدید برای نسخه‌های قدیمی‌تر (به‌صورت Best-effort)
 */
function ensureConfigUsageCacheColumns(): void {
    try {
        $pdo = db();

        $cols = [];
        $q = $pdo->query("SHOW COLUMNS FROM config_usage_cache");
        if ($q) {
            while ($r = $q->fetch()) {
                $cols[] = $r['Field'] ?? $r[0] ?? null;
            }
        }

        if (!in_array('delivered_at', $cols, true)) {
            $pdo->exec("ALTER TABLE config_usage_cache ADD COLUMN delivered_at BIGINT NULL AFTER last_checked");
        }
        if (!in_array('no_use_notice_sent', $cols, true)) {
            $pdo->exec("ALTER TABLE config_usage_cache ADD COLUMN no_use_notice_sent TINYINT(1) NOT NULL DEFAULT 0 AFTER delivered_at");
        }
        if (!in_array('warn80_sent', $cols, true)) {
            $pdo->exec("ALTER TABLE config_usage_cache ADD COLUMN warn80_sent TINYINT(1) NOT NULL DEFAULT 0 AFTER no_use_notice_sent");
        }
        if (!in_array('warn10_sent', $cols, true)) {
            $pdo->exec("ALTER TABLE config_usage_cache ADD COLUMN warn10_sent TINYINT(1) NOT NULL DEFAULT 0 AFTER warn80_sent");
        }
        if (!in_array('expired_sent', $cols, true)) {
            $pdo->exec("ALTER TABLE config_usage_cache ADD COLUMN expired_sent TINYINT(1) NOT NULL DEFAULT 0 AFTER warn10_sent");
        }
        if (!in_array('last_warned_at', $cols, true)) {
            $pdo->exec("ALTER TABLE config_usage_cache ADD COLUMN last_warned_at BIGINT NULL AFTER expired_sent");
        }
        if (!in_array('meta', $cols, true)) {
            $pdo->exec("ALTER TABLE config_usage_cache ADD COLUMN meta JSON NULL AFTER last_warned_at");
        }

        // ایندکس کمکی برای جستجوی یادآوری‌ها
        $idx = [];
        $qi = $pdo->query("SHOW INDEX FROM config_usage_cache");
        if ($qi) {
            while ($r = $qi->fetch()) {
                $idx[] = $r['Key_name'] ?? $r[2] ?? null;
            }
        }
        if (!in_array('idx_delivered_notice', $idx, true)) {
            $pdo->exec("ALTER TABLE config_usage_cache ADD INDEX idx_delivered_notice (delivered_at, no_use_notice_sent)");
        }
    } catch (Throwable $e) {
        log_error('ensureConfigUsageCacheColumns failed', ['msg' => $e->getMessage()]);
    }
}


function markConfigDelivered(int $configId): void {
    ensureConfigUsageCacheTable();
    $pdo = db();
    $now = time();

    // اگر رکورد کش وجود نداشته باشد ایجاد می‌کنیم، و اگر وجود داشته باشد delivered_at را آپدیت می‌کنیم
    $sql = "INSERT INTO config_usage_cache (config_id, delivered_at, no_use_notice_sent, warn80_sent, warn10_sent, last_warned_at)\n"
         . "VALUES (:cid, :d, 0, 0, 0, NULL)\n"
         . "ON DUPLICATE KEY UPDATE delivered_at=VALUES(delivered_at), no_use_notice_sent=0, warn80_sent=0, warn10_sent=0, last_warned_at=NULL";
    $st = $pdo->prepare($sql);
    $st->execute([':cid' => (int)$configId, ':d' => $now]);
}

/**
 * درخواست HTTP برای لینک ساب/پنل و دریافت هدرها و بدنه
 * @return array{ok:bool,status:int,headers:array,body:string,err?:string}
 */
function httpGetWithHeaders(string $url, int $timeout = 25): array {
    $headers = [];
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 5,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36',
        // پاسخ‌های gzip/br را هم باز کند
        CURLOPT_ENCODING => '',
        // هدرهای معمول مرورگر برای جلوگیری از بلاک شدن برخی WAF ها
        CURLOPT_HTTPHEADER => [
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: fa-IR,fa;q=0.9,en-US;q=0.7,en;q=0.6',
            'Cache-Control: no-cache',
            'Pragma: no-cache',
        ],
        CURLOPT_HEADERFUNCTION => function($ch, $headerLine) use (&$headers) {
            $len = strlen($headerLine);
            $parts = explode(':', $headerLine, 2);
            if (count($parts) === 2) {
                $name = strtolower(trim($parts[0]));
                $value = trim($parts[1]);
                if ($name !== '') {
                    // ممکن است یک هدر چندبار بیاید
                    if (!isset($headers[$name])) $headers[$name] = $value;
                }
            }
            return $len;
        },
    ]);

    $body = curl_exec($ch);
    $err = null;
    if ($body === false) {
        $err = curl_error($ch);
        $body = '';
    }
    $status = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // بعضی هاست‌ها برای برخی دامنه‌ها مشکل SSL/CA دارند؛ برای userdatalink یک بار با SSL verify خاموش retry می‌کنیم.
    if ($err !== null && stripos($url, 'userdatalink.com') !== false) {
        if (stripos($err, 'ssl') !== false || stripos($err, 'certificate') !== false) {
            $headers = [];
            $ch2 = curl_init();
            curl_setopt_array($ch2, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_MAXREDIRS => 5,
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_TIMEOUT => $timeout,
                CURLOPT_USERAGENT => 'Mozilla/5.0 (compatible; DNetBot/1.0)',
                CURLOPT_ENCODING => '',
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => 0,
                CURLOPT_HTTPHEADER => [
                    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                    'Accept-Language: fa-IR,fa;q=0.9,en-US;q=0.7,en;q=0.6',
                    'Cache-Control: no-cache',
                    'Pragma: no-cache',
                ],
                CURLOPT_HEADERFUNCTION => function($ch, $headerLine) use (&$headers) {
                    $len = strlen($headerLine);
                    $parts = explode(':', $headerLine, 2);
                    if (count($parts) === 2) {
                        $name = strtolower(trim($parts[0]));
                        $value = trim($parts[1]);
                        if ($name !== '') {
                            if (!isset($headers[$name])) $headers[$name] = $value;
                        }
                    }
                    return $len;
                },
            ]);
            $body2 = curl_exec($ch2);
            $err2 = null;
            if ($body2 === false) {
                $err2 = curl_error($ch2);
                $body2 = '';
            }
            $status2 = (int)curl_getinfo($ch2, CURLINFO_HTTP_CODE);
            curl_close($ch2);

            // اگر retry موفق بود، همان را برگردان
            if ($err2 === null && $status2 >= 200 && $status2 < 400) {
                return [
                    'ok' => true,
                    'status' => $status2,
                    'headers' => $headers,
                    'body' => (string)$body2,
                ];
            }
        }
    }

    return [
        'ok' => ($err === null && $status >= 200 && $status < 400),
        'status' => $status,
        'headers' => $headers,
        'body' => (string)$body,
        'err' => $err,
    ];
}


/**
 * درخواست POST با payload JSON و دریافت هدرها و بدنه (برای API های پنل‌ها)
 * @return array{ok:bool,status:int,headers:array,body:string,err?:string}
 */
function httpPostJsonWithHeaders(string $url, array $payload, int $timeout = 25, array $extraHeaders = []): array {
    $headers = [];
    $ch = curl_init();
    $baseHeaders = [
        'Accept: application/json, text/plain, */*',
        'Content-Type: application/json',
        // برای جلوگیری از zstd (ممکن است روی بعضی سرورها curl نتواند decode کند)
        'Accept-Encoding: gzip, deflate',
        'Accept-Language: fa-IR,fa;q=0.9,en-US;q=0.7,en;q=0.6',
        'Cache-Control: no-cache',
        'Pragma: no-cache',
    ];
    $allHeaders = array_merge($baseHeaders, $extraHeaders);

    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 3,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36',
        CURLOPT_ENCODING => '', // اگر gzip باشد، curl باز می‌کند
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_UNICODE),
        CURLOPT_HTTPHEADER => $allHeaders,
        CURLOPT_HEADERFUNCTION => function($ch, $headerLine) use (&$headers) {
            $len = strlen($headerLine);
            $parts = explode(':', $headerLine, 2);
            if (count($parts) === 2) {
                $name = strtolower(trim($parts[0]));
                $value = trim($parts[1]);
                if ($name !== '' && !isset($headers[$name])) {
                    $headers[$name] = $value;
                }
            }
            return $len;
        },
    ]);

    $body = curl_exec($ch);
    $err = null;
    if ($body === false) {
        $err = curl_error($ch);
        $body = '';
    }
    $status = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    return [
        'ok' => ($err === null && $status >= 200 && $status < 400),
        'status' => $status,
        'headers' => $headers,
        'body' => (string)$body,
        'err' => $err,
    ];
}

/**
 * پنل userdatalink.com: اطلاعات مصرف/انقضا از API خارجی adspeedo می‌آید.
 * طبق HAR: POST https://api1.adspeedo.com/services/info  {token: "..."}
 */
function fetchUsageFromAdsSpeedo(string $token): array {
    $token = trim($token);
    if ($token === '') {
        return ['ok'=>false,'used_bytes'=>0,'total_bytes'=>0,'expire_ts'=>null,'source'=>'adspeedo','err'=>'empty_token'];
    }

    $api = 'https://api1.adspeedo.com/services/info';
    $r = httpPostJsonWithHeaders($api, ['token' => $token], 25, [
        // شبیه مرورگر (در HAR Origin=userdatalink.com)
        'Origin: https://userdatalink.com',
        'Referer: https://userdatalink.com/',
    ]);

    if (empty($r['ok'])) {
        return ['ok'=>false,'used_bytes'=>0,'total_bytes'=>0,'expire_ts'=>null,'source'=>'adspeedo','err'=>$r['err'] ?? 'http_fail'];
    }

    $body = trim((string)($r['body'] ?? ''));
    $js = json_decode($body, true);
    if (!is_array($js) || empty($js['service']) || !is_array($js['service'])) {
        return ['ok'=>false,'used_bytes'=>0,'total_bytes'=>0,'expire_ts'=>null,'source'=>'adspeedo','err'=>'bad_json'];
    }

    $svc = $js['service'];

    // طبق نمونه: transfer_enablegb = کل حجم (GB) ، transfer_totalgb = مصرف (GB)
    $totalGb = null;
    if (isset($svc['transfer_enablegb']) && is_numeric($svc['transfer_enablegb'])) $totalGb = (float)$svc['transfer_enablegb'];
    $usedGb = null;
    if (isset($svc['transfer_totalgb']) && is_numeric($svc['transfer_totalgb'])) $usedGb = (float)$svc['transfer_totalgb'];

    // fallback: ugb+dgb
    if ($usedGb === null) {
        $u = (isset($svc['ugb']) && is_numeric($svc['ugb'])) ? (float)$svc['ugb'] : 0.0;
        $d = (isset($svc['dgb']) && is_numeric($svc['dgb'])) ? (float)$svc['dgb'] : 0.0;
        if (($u+$d) > 0) $usedGb = $u + $d;
    }

    $expireTs = null;
    if (isset($svc['expired_at']) && is_numeric($svc['expired_at'])) {
        $expireTs = (int)$svc['expired_at'];
        if ($expireTs <= 0) $expireTs = null;
    } elseif (isset($svc['remainDays']) && is_numeric($svc['remainDays'])) {
        $expireTs = time() + ((int)$svc['remainDays'] * 86400);
    }

    if ($totalGb === null || $usedGb === null || $totalGb <= 0) {
        return ['ok'=>false,'used_bytes'=>0,'total_bytes'=>0,'expire_ts'=>null,'source'=>'adspeedo','err'=>'missing_fields'];
    }

    $totalBytes = (int)round($totalGb * 1073741824);
    $usedBytes = (int)max(0, round($usedGb * 1073741824));

    return [
        'ok' => true,
        'used_bytes' => $usedBytes,
        'total_bytes' => $totalBytes,
        'expire_ts' => $expireTs,
        'source' => 'adspeedo',
        'raw' => [
            'service_id' => $svc['id'] ?? null,
            'email' => $svc['email'] ?? null,
            'status' => $svc['status'] ?? null,
            'remainDays' => $svc['remainDays'] ?? null,
            'usedpercent' => $svc['usedpercent'] ?? null,
        ],
    ];
}

/**
 * تبدیل اعداد فارسی/عربی به انگلیسی و پاکسازی جداکننده‌ها
 */
function normalizeFaDigits(string $s): string {
    $map = [
        '۰'=>'0','۱'=>'1','۲'=>'2','۳'=>'3','۴'=>'4','۵'=>'5','۶'=>'6','۷'=>'7','۸'=>'8','۹'=>'9',
        '٠'=>'0','١'=>'1','٢'=>'2','٣'=>'3','٤'=>'4','٥'=>'5','٦'=>'6','٧'=>'7','٨'=>'8','٩'=>'9',
    ];
    $s = strtr($s, $map);
    // جداکننده‌های رایج فارسی
    $s = str_replace(["\xE2\x80\x8C", '٬', '،'], [ '', ',', ',' ], $s); // ZWNJ + thousands/comma variants
    $s = str_replace(['٫'], ['.'], $s);
    return $s;
}

/**
 * تبدیل رشته عددی (ممکن است شامل , یا . باشد) به float قابل محاسبه
 */
function parseFlexibleNumber(string $raw): float {
    $raw = trim(normalizeFaDigits($raw));
    // اگر فقط یک comma داریم و نقطه نداریم، احتمالاً comma نقش اعشار دارد (مثل 1,671)
    if (strpos($raw, '.') === false && substr_count($raw, ',') === 1) {
        [$a, $b] = explode(',', $raw, 2);
        if (strlen($b) <= 3) {
            $raw = $a . '.' . $b;
        } else {
            $raw = $a . $b; // هزارگان
        }
    }
    // سایر comma ها را به عنوان هزارگان حذف می‌کنیم
    $raw = str_replace(',', '', $raw);
    // فقط کاراکترهای عدد/نقطه/منفی باقی بماند
    $raw = preg_replace('/[^0-9.\-]/', '', $raw);
    if ($raw === '' || $raw === '-' || $raw === '.') return 0.0;
    return (float)$raw;
}

/**
 * تبدیل مقدار+واحد به بایت
 */
function numberUnitToBytes(float $val, string $unit): int {
    $u = mb_strtolower(trim($unit));
    $mul = 1;
    if (strpos($u, 'gb') !== false || strpos($u, 'گیگ') !== false) {
        $mul = 1073741824;
    } elseif (strpos($u, 'mb') !== false || strpos($u, 'مگ') !== false) {
        $mul = 1048576;
    } elseif (strpos($u, 'kb') !== false || strpos($u, 'کیلو') !== false) {
        $mul = 1024;
    } else {
        $mul = 1;
    }
    return (int)max(0, round($val * $mul));
}

/**
 * پارس کردن Subscription-Userinfo (استاندارد رایج پنل‌ها)
 * نمونه: upload=123; download=456; total=10737418240; expire=1733333333
 */
function parseSubscriptionUserinfo(string $val): array {
    $out = [];
    foreach (explode(';', $val) as $part) {
        $part = trim($part);
        if ($part === '') continue;
        $kv = explode('=', $part, 2);
        if (count($kv) !== 2) continue;
        $k = strtolower(trim($kv[0]));
        $v = trim($kv[1]);
        if ($k === '') continue;
        if (is_numeric($v)) {
            $out[$k] = (int)$v;
        } else {
            $out[$k] = $v;
        }
    }
    return $out;
}

/**
 * نرمال‌سازی timestamp انقضا
 * - برخی پنل‌ها مقدار expire را به میلی‌ثانیه/میکروثانیه می‌دهند.
 * - برخی هم تاریخ را به صورت رشته (ISO/...) می‌فرستند.
 */
function normalizeExpireUnixTs($v): ?int {
    if ($v === null) return null;

    // اگر رشته است
    if (is_string($v)) {
        $v = trim($v);
        if ($v === '') return null;
        if (is_numeric($v)) {
            $v = (int)$v;
        } else {
            // تلاش برای پارس تاریخ
            $ts = strtotime($v);
            if ($ts !== false) return (int)$ts;
            $ts2 = parseDateToTimestamp($v);
            return $ts2 ? (int)$ts2 : null;
        }
    }

    if (is_float($v)) $v = (int)$v;
    if (!is_int($v)) return null;
    if ($v <= 0) return null;

    // میلی‌ثانیه/میکروثانیه => تبدیل به ثانیه
    while ($v > 20000000000) { // بزرگ‌تر از سال ~2600 در ثانیه
        $v = (int)floor($v / 1000);
    }

    return ($v > 0) ? (int)$v : null;
}

/**
 * پارس اطلاعات پنل zerofee.ok-exor.ir از HTML سرور-ساید رندر شده
 * خروجی: used_bytes, total_bytes, days_left, expire_ts(optional)
 */
function parseOkExorHtmlUsage(string $html): array {
    $out = ['ok' => false];

    // نرمال‌سازی نیم‌فاصله‌ها و اعداد فارسی
    $htmlNorm = str_replace(["\xE2\x80\x8C", "\xE2\x80\x8F"], ' ', $html); // ZWNJ / RLM
    // تبدیل ارقام فارسی به انگلیسی
    $fa = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹','٠','١','٢','٣','٤','٥','٦','٧','٨','٩'];
    $en = ['0','1','2','3','4','5','6','7','8','9','0','1','2','3','4','5','6','7','8','9'];
    $htmlNorm = str_replace($fa, $en, $htmlNorm);

    // 1) استخراج usedTraffic و dataLimit از x-data
    $usedBytes = null;
    $totalBytes = null;
    if (preg_match('/x-data\s*=\s*"\{\s*usedTraffic\s*:\s*([0-9]+)\s*,\s*dataLimit\s*:\s*([0-9]+)\s*\}"/u', $htmlNorm, $m)) {
        $usedBytes = (int)$m[1];
        $totalBytes = (int)$m[2];
    } else {
        // حالت‌هایی که اسپیس/فاصله/ترتیب متفاوت باشد
        if (preg_match('/usedTraffic\s*:\s*([0-9]+)\s*,\s*dataLimit\s*:\s*([0-9]+)/u', $htmlNorm, $m)) {
            $usedBytes = (int)$m[1];
            $totalBytes = (int)$m[2];
        }
    }

    // 2) استخراج days_left از بخش remainingDays (مثل: <span>(17)</span>)
    $daysLeft = null;
    if (preg_match('/x-text\s*=\s*"\$t\(\'remainingDays\'\)".{0,800}?<span>\s*\(\s*([0-9]+)\s*\)\s*<\/span>/su', $htmlNorm, $m)) {
        $daysLeft = (int)$m[1];
    } else {
        // fallback: هر جا "(عدد)" بعد از remainingDays آمده باشد
        if (preg_match('/remainingDays.{0,800}?\(\s*([0-9]+)\s*\)/su', $htmlNorm, $m)) {
            $daysLeft = (int)$m[1];
        }
    }

    // 3) استخراج تاریخ پایان اگر در متغیر expireDateVar باشد (YYYY-MM-DD HH:MM:SS)
    $expireTs = null;
    if (preg_match('/expireDateVar\s*[:=]\s*[\'"]([0-9]{4}\-[0-9]{2}\-[0-9]{2}\s+[0-9]{2}:[0-9]{2}:[0-9]{2})[\'"]/u', $htmlNorm, $m)) {
        $ts = strtotime($m[1]);
        if ($ts !== false) $expireTs = (int)$ts;
    }

    if ($usedBytes !== null && $totalBytes !== null && $daysLeft !== null) {
        $out = [
            'ok' => true,
            'used_bytes' => max(0, $usedBytes),
            'total_bytes' => max(0, $totalBytes),
            'expire_ts' => $expireTs,
            'days_left' => $daysLeft,
            'source' => 'okexor-html',
            'raw' => [
                'usedTraffic' => $usedBytes,
                'dataLimit' => $totalBytes,
                'days_left' => $daysLeft,
                'expire_ts' => $expireTs,
            ],
        ];
    }

    return $out;
}



function bytesToGbFloat(int $bytes): float {
    if ($bytes <= 0) return 0.0;
    return $bytes / 1073741824.0; // 1024^3
}

function formatGb(int $bytes): string {
    $gb = bytesToGbFloat($bytes);
    if ($gb >= 10) return number_format($gb, 1) . ' GB';
    return number_format($gb, 2) . ' GB';
}

/**
 * استخراج مصرف از صفحه/هدر/متن
 * @return array{ok:bool,used_bytes:int,total_bytes:int,expire_ts:?int,source:string,raw?:array,err?:string}
 */
function fetchConfigUsage(string $link): array {
    $link = trim($link);
    // fragment (#...) به سرور ارسال نمی‌شود
    $link = preg_replace('/#.*/u', '', $link);

    if ($link === '') {
        return ['ok' => false, 'used_bytes' => 0, 'total_bytes' => 0, 'expire_ts' => null, 'source' => 'empty', 'err' => 'empty_link'];
    }

    // --- ساخت لیست لینک‌های جایگزین (برای پنل‌هایی مثل userdatalink که صفحه /q?token= دارند)
    $tryLinks = [$link];

    $u = @parse_url($link);
    $host = strtolower((string)($u['host'] ?? ''));
    $scheme = (string)($u['scheme'] ?? 'https');
    $path = (string)($u['path'] ?? '');
    $query = (string)($u['query'] ?? '');

    $token = null;
    if ($query !== '') {
        parse_str($query, $qs);
        if (!empty($qs['token']) && is_string($qs['token'])) {
            $token = trim($qs['token']);
        }
    }

    if ($token && $host !== '') {
        $base = $scheme . '://' . $host;

        // اگر لینک از نوع /q?token= بود، چند مسیر رایج ساب/ای‌پی‌آی را هم امتحان می‌کنیم
        if (preg_match('~^/q/?$~i', $path) || stripos($path, '/q') === 0) {
            $tryLinks[] = $base . '/sub/' . rawurlencode($token);
            $tryLinks[] = $base . '/sub?token=' . rawurlencode($token);
            $tryLinks[] = $base . '/api/sub/' . rawurlencode($token);
            $tryLinks[] = $base . '/api/sub?token=' . rawurlencode($token);
            $tryLinks[] = $base . '/subscribe/' . rawurlencode($token);
            $tryLinks[] = $base . '/subscription/' . rawurlencode($token);
        }
    }

    // یکتا سازی
    $tryLinks = array_values(array_unique(array_filter(array_map('trim', $tryLinks))));

    $bestErr = null;

    // دامنه‌ای که روزهای باقی‌مانده را از UI نمایش می‌دهد و ممکن است با تاریخ پایان متفاوت باشد
    $isOkExorPanel = ($host !== '' && stripos($host, 'zerofee.ok-exor.ir') !== false && stripos($path, '/sub/') === 0);


    // --- userdatalink.com/q?token= (طبق HAR: API خارجی adspeedo با POST)
    if ($token && $host !== '' && (stripos($host, 'userdatalink.com') !== false) && (stripos($path, '/q') === 0)) {
        $ads = fetchUsageFromAdsSpeedo($token);
        if (!empty($ads['ok'])) {
            return $ads;
        }
        if (!empty($ads['err'])) {
            $bestErr = $ads['err'];
        }
        // اگر adspeedo جواب نداد، ادامه می‌دهیم (fallback به روش‌های دیگر)
    }

    foreach ($tryLinks as $tryUrl) {
        $res = httpGetWithHeaders($tryUrl, 25);
        if (empty($res['ok'])) {
            $bestErr = $res['err'] ?? 'http_failed';
            continue;
        }

        $headers = $res['headers'] ?? [];
        $body = (string)($res['body'] ?? '');

        // 1) هدر استاندارد Subscription-Userinfo (بهترین منبع)
        $userinfo = null;
        if (isset($headers['subscription-userinfo'])) {
            $userinfo = $headers['subscription-userinfo'];
        } elseif (isset($headers['Subscription-Userinfo'])) {
            $userinfo = $headers['Subscription-Userinfo'];
        } elseif (isset($headers['subscription-userinfo' . "\r"])) {
            $userinfo = $headers['subscription-userinfo' . "\r"];
        }

        $prefUsed = null;
        $prefTotal = null;
        $prefExpire = null;
        $prefRaw = null;

        if ($userinfo !== null && trim($userinfo) !== '') {
            $p = parseSubscriptionUserinfo($userinfo);
            $upload = (int)($p['upload'] ?? 0);
            $download = (int)($p['download'] ?? 0);
            $total = (int)($p['total'] ?? 0);
            $expire = normalizeExpireUnixTs($p['expire'] ?? ($p['expires'] ?? null));
            $used = max(0, $upload + $download);

            $prefUsed = $used;
            $prefTotal = max(0, $total);
            $prefExpire = ($expire && $expire > 0) ? $expire : null;
            $prefRaw = $p;

            // در اکثر پنل‌ها همین هدر کافی است، اما برای پنل zerofee.ok-exor.ir باید روزهای باقی‌مانده را از HTML بخوانیم
            if (!$isOkExorPanel && $prefExpire !== null) {
                return [
                    'ok' => true,
                    'used_bytes' => $prefUsed,
                    'total_bytes' => $prefTotal,
                    'expire_ts' => $prefExpire,
                    'days_left' => computeDaysLeftFromExpire($prefExpire),
                    'source' => 'subscription-userinfo',
                    'raw' => $prefRaw,
                ];
            }
        }

        // 1.4) پنل zerofee.ok-exor.ir: اطلاعات به صورت SSR داخل HTML است (بدون API جداگانه)
        if ($isOkExorPanel) {
            $parsed = parseOkExorHtmlUsage($body);
            if (!empty($parsed['ok'])) {
                // اگر از هدر subscription-userinfo هم چیزی داشتیم، در raw نگه می‌داریم
                if ($prefRaw !== null) {
                    $parsed['raw']['subscription_userinfo'] = $prefRaw;
                }
                return $parsed;
            }
        }

        // 1.5) اگر JSON بود (برخی پنل‌ها)
        $btrim = trim($body);
        if ($btrim !== '' && (((isset($btrim[0]) && $btrim[0] === '{') || (isset($btrim[0]) && $btrim[0] === '[')))) {
            $js = json_decode($btrim, true);
            if (is_array($js)) {
                // چند نام رایج
                $usedBytes = null;
                $totalBytes = null;
                $expireTs = null;

                foreach (['used_bytes','used','usage','traffic_used'] as $k) {
                    if (isset($js[$k]) && is_numeric($js[$k])) { $usedBytes = (int)$js[$k]; break; }
                }
                foreach (['total_bytes','total','quota','traffic_total'] as $k) {
                    if (isset($js[$k]) && is_numeric($js[$k])) { $totalBytes = (int)$js[$k]; break; }
                }
                foreach (['expire','expire_ts','expires_at','expires','expired_at','expireAt'] as $k) {
                    if (isset($js[$k])) {
                        $rawExp = $js[$k];
                        if (is_numeric($rawExp) || is_string($rawExp)) {
                            $tmpTs = normalizeExpireUnixTs($rawExp);
                            if ($tmpTs) { $expireTs = (int)$tmpTs; break; }
                        }
                    }
                }

                if ($usedBytes !== null && $totalBytes !== null) {
                    return [
                        'ok' => true,
                        'used_bytes' => max(0, (int)$usedBytes),
                        'total_bytes' => max(0, (int)$totalBytes),
                        'expire_ts' => ($expireTs && $expireTs > 0) ? (int)$expireTs : null,
                        'source' => 'json',
                        'raw' => $js,
                    ];
                }
            }
        }

        // 2) بعضی پنل‌ها/ساب‌ها خروجی را base64 می‌دهند (مثل برخی /sub ها)
        $decoded = null;
        if ($btrim !== '' && strlen($btrim) > 60 && preg_match('/^[A-Za-z0-9+\/\r\n=]+$/', $btrim)) {
            $compact = preg_replace('/\s+/', '', $btrim);
            if (strlen($compact) >= 80) {
                $try = base64_decode($compact, true);
                if ($try !== false && is_string($try)) {
                    if (
                        stripos($try, 'vless://') !== false ||
                        stripos($try, 'vmess://') !== false ||
                        stripos($try, 'trojan://') !== false ||
                        stripos($try, 'ss://') !== false ||
                        preg_match('/\bGB\b|\bMB\b|Days|روز/u', $try)
                    ) {
                        $decoded = $try;
                    }
                }
            }
        }

        $candidates = [$body];
        if ($decoded !== null) $candidates[] = $decoded;

        // --- استخراج از متن (HTML یا Plain)
        $extract = function(string $text) {
            $stripped = trim(preg_replace('/\s+/', ' ', strip_tags($text)));
            $stripped_norm = normalizeFaDigits($stripped);

            $best = null;
            $unitMul = ['B'=>1,'KB'=>1024,'MB'=>1048576,'GB'=>1073741824];

            // 2.a) "21.88 GB / 30.0 GB"
            if (preg_match_all('/([0-9]+(?:\.[0-9]+)?)\s*(B|KB|MB|GB)\s*\/\s*([0-9]+(?:\.[0-9]+)?)\s*(B|KB|MB|GB)/i', $stripped_norm, $all, PREG_SET_ORDER)) {
                foreach ($all as $m) {
                    $usedVal = (float)$m[1];
                    $usedUnit = strtoupper($m[2]);
                    $totalVal = (float)$m[3];
                    $totalUnit = strtoupper($m[4]);

                    $usedBytes = (int)round($usedVal * ($unitMul[$usedUnit] ?? 1));
                    $totalBytes = (int)round($totalVal * ($unitMul[$totalUnit] ?? 1));
                    if ($totalBytes > 0 && ($best === null || $totalBytes > ($best['total'] ?? 0))) {
                        $best = ['used' => max(0, $usedBytes), 'total' => max(0, $totalBytes)];
                    }
                }
            }

            // 2.b) الگوی فارسی (پنل‌های شبیه userdatalink)
            if ($best === null) {
                $u = null; $t = null;

                // مصرف
                if (preg_match('/(?:حجم|ترافیک)\s*مصرفی(?:\s*شما)?\D*([0-9][0-9\.,]+)\s*(گیگابایت|گیگ|GB|مگابایت|مگ|MB|کیلوبایت|کیلو|KB|بایت|B)/u', $stripped_norm, $um)) {
                    $uVal = parseFlexibleNumber($um[1]);
                    $u = numberUnitToBytes($uVal, $um[2]);
                }
                // کل
                if (preg_match('/(?:حجم|ترافیک)\s*(?:اولیه|کل|سقف)(?:\s*شما)?\D*([0-9][0-9\.,]+)\s*(گیگابایت|گیگ|GB|مگابایت|مگ|MB|کیلوبایت|کیلو|KB|بایت|B)/u', $stripped_norm, $tm)) {
                    $tVal = parseFlexibleNumber($tm[1]);
                    $t = numberUnitToBytes($tVal, $tm[2]);
                }

                if ($u !== null && $t !== null && $t > 0) {
                    $best = ['used' => (int)$u, 'total' => (int)$t];
                }
            }

            // days left
            $daysLeft = null;
            if (preg_match('/روز(?:های)?\s*باقی(?:‌|\s)*مانده\D*\(?\s*([0-9]{1,4})\s*\)?/u', $stripped_norm, $dm)) {
                $daysLeft = (int)$dm[1];
            } elseif (preg_match('/زمان\s*باقی\s*مانده\D*\(?\s*([0-9]{1,4})\s*\)?\s*روز/u', $stripped_norm, $dm2)) {
                $daysLeft = (int)$dm2[1];
            } elseif (preg_match('/\(?\s*([0-9]{1,4})\s*\)?\s*روز/u', $stripped_norm, $dm3)) {
                $daysLeft = (int)$dm3[1];
            } elseif (preg_match('/\b([0-9]{1,4})\s*Days\b/i', $stripped_norm, $dm4)) {
                $daysLeft = (int)$dm4[1];
            }

            if ($best !== null) {
                $expireTs = null;

// تلاش برای خواندن تاریخ پایان/انقضا (اولویت با تاریخ دقیق)
if (preg_match('/تاریخ\s*(?:پایان|انقضا)\s*(?:اشتراک|سرویس)?\s*[:：]?\s*([0-9]{4}[\/\-][0-9]{1,2}[\/\-][0-9]{1,2}(?:[ T][0-9]{1,2}:[0-9]{2}(?::[0-9]{2})?)?)/u', $stripped_norm, $em)) {
    $expireTs = parseDateToTimestamp($em[1], false);
}
if (!$expireTs && $daysLeft !== null) {
    // اگر تاریخ دقیق نبود، از روزهای باقی‌مانده تخمین می‌زنیم
    $expireTs = time() + ((int)$daysLeft * 86400);
}
                $best['days_left'] = ($daysLeft !== null) ? $daysLeft : computeDaysLeftFromExpire($expireTs);
                $best['expire_ts'] = $expireTs;
                return $best;
            }
            return null;
        };

        $bestOverall = null;
        foreach ($candidates as $cand) {
            $r = $extract((string)$cand);
            if ($r && !empty($r['total'])) {
                if ((int)$r['total'] < 1024 * 1024) continue;
                if ($bestOverall === null || (int)$r['total'] > (int)($bestOverall['total'] ?? 0)) {
                    $bestOverall = $r;
                }
            }
        }

        if ($bestOverall !== null || ($prefTotal !== null && $prefTotal > 0)) {
            $used = ($prefUsed !== null) ? (int)$prefUsed : (int)($bestOverall['used'] ?? 0);
            $total = ($prefTotal !== null) ? (int)$prefTotal : (int)($bestOverall['total'] ?? 0);

            // expire: اگر از هدر یا HTML/JSON بدست آمد نگه می‌داریم، ولی نمایش روزها را از days_left می‌گیریم
            $expireTs = $bestOverall['expire_ts'] ?? null;
            if (!$expireTs && $prefExpire) $expireTs = $prefExpire;

            // days_left: اگر متن پنل مقدار دقیقی داد، همان را به عنوان مرجع ذخیره/نمایش استفاده می‌کنیم
            $daysLeft = $bestOverall['days_left'] ?? null;
            if ($daysLeft === null && $expireTs) $daysLeft = computeDaysLeftFromExpire((int)$expireTs);

            return [
                'ok' => true,
                'used_bytes' => max(0, (int)$used),
                'total_bytes' => max(0, (int)$total),
                'expire_ts' => ($expireTs && $expireTs > 0) ? (int)$expireTs : null,
                'days_left' => ($daysLeft !== null) ? (int)$daysLeft : null,
                'source' => ($prefUsed !== null && $bestOverall !== null) ? 'mixed' : (($prefUsed !== null) ? 'subscription-userinfo' : 'html/text'),
                'raw' => ['url' => $tryUrl],
            ];
        }

        // 3) بعضی پنل‌ها (مثل userdatalink.com/q) داده را با JS از API می‌گیرند و داخل HTML خام نیست.
        // تلاش می‌کنیم از HTML/اسکریپت‌ها، آدرس API را کشف کنیم و داده را از همان بخوانیم.
        if ($token && $host !== '' && (stripos($host, 'userdatalink.com') !== false) && (stripos($path, '/q') === 0 || stripos($tryUrl, '/q?') !== false)) {
            $disc = discoverUsageViaFrontend($scheme.'://'.$host, $token, $body);
            if (!empty($disc['ok'])) {
                return $disc;
            }
            if (!empty($disc['err'])) {
                $bestErr = $disc['err'];
            }
        }
    }

    return ['ok' => false, 'used_bytes' => 0, 'total_bytes' => 0, 'expire_ts' => null, 'source' => 'parse_failed', 'err' => $bestErr ?? 'parse_failed'];
}

/**
 * کشف API مصرف/انقضا از HTML/Bundle های فرانت و خواندن اطلاعات.
 * @return array{ok:bool,used_bytes:int,total_bytes:int,expire_ts:?int,source:string,raw?:array,err?:string}
 */
function discoverUsageViaFrontend(string $base, string $token, string $html): array {
    $base = rtrim($base, '/');
    $token = trim($token);
    if ($base === '' || $token === '' || $html === '') {
        return ['ok' => false, 'used_bytes' => 0, 'total_bytes' => 0, 'expire_ts' => null, 'source' => 'frontend', 'err' => 'frontend_empty'];
    }

    // --- کش ساده روی فایل (برای جلوگیری از دانلود دائم bundle ها)
    $cacheFile = __DIR__ . '/._frontend_api_cache.json';
    $cache = [];
    if (is_file($cacheFile)) {
        $tmp = @json_decode((string)@file_get_contents($cacheFile), true);
        if (is_array($tmp)) $cache = $tmp;
    }
    $cacheKey = parse_url($base, PHP_URL_HOST) ?: $base;
    $now = time();
    $templates = [];

    if (isset($cache[$cacheKey]) && is_array($cache[$cacheKey]) && !empty($cache[$cacheKey]['ts']) && ($now - (int)$cache[$cacheKey]['ts'] < 7*86400)) {
        $templates = is_array($cache[$cacheKey]['templates'] ?? null) ? $cache[$cacheKey]['templates'] : [];
    }

    if (empty($templates)) {
        $templates = array_values(array_unique(array_filter(discoverApiTemplatesFromHtmlAndScripts($html, $base))));
        // ذخیره کش
        $cache[$cacheKey] = ['ts' => $now, 'templates' => $templates];
        @file_put_contents($cacheFile, json_encode($cache, JSON_UNESCAPED_UNICODE));
    }

    if (empty($templates)) {
        return ['ok' => false, 'used_bytes' => 0, 'total_bytes' => 0, 'expire_ts' => null, 'source' => 'frontend', 'err' => 'no_templates'];
    }

    // تبدیل template به URL واقعی
    $tryUrls = [];
    foreach ($templates as $tpl) {
        $tpl = trim((string)$tpl);
        if ($tpl === '') continue;

        // جایگزینی placeholder
        if (strpos($tpl, '{token}') !== false) {
            $tryUrls[] = str_replace('{token}', rawurlencode($token), $tpl);
            continue;
        }
        // اگر فقط "token=" داشت
        if (preg_match('/token=\s*$/i', $tpl)) {
            $tryUrls[] = $tpl . rawurlencode($token);
            continue;
        }
        // اگر توکن داخل URL نبود ولی token= داشت
        if (stripos($tpl, 'token=') !== false && stripos($tpl, $token) === false) {
            // تلاش برای افزودن value اگر خالی است
            $tryUrls[] = preg_replace('/(token=)([^&"\']*)/i', '$1' . rawurlencode($token), $tpl, 1);
            continue;
        }
        // مسیرهایی که احتمالاً توکن را در path می‌گیرند
        if (substr($tpl, -1) === '/') {
            $tryUrls[] = $tpl . rawurlencode($token);
            continue;
        }

        // fallback: اگر template یک مسیر بود، هم به صورت ?token= هم /token امتحان می‌کنیم
        if (strpos($tpl, '?') === false) {
            $tryUrls[] = $tpl . '?token=' . rawurlencode($token);
            $tryUrls[] = $tpl . '/' . rawurlencode($token);
        } else {
            $tryUrls[] = $tpl;
        }
    }

    $tryUrls = array_values(array_unique(array_filter($tryUrls)));

    foreach ($tryUrls as $u) {
        $r = httpGetWithHeaders($u, 25);
        if (empty($r['ok'])) continue;
        $parsed = parseUsageFromHeadersBody($r['headers'] ?? [], (string)($r['body'] ?? ''), $u);
        if (!empty($parsed['ok'])) {
            $parsed['source'] = 'frontend_api';
            $parsed['raw'] = array_merge($parsed['raw'] ?? [], ['api_url' => $u]);
            return $parsed;
        }
    }

    return ['ok' => false, 'used_bytes' => 0, 'total_bytes' => 0, 'expire_ts' => null, 'source' => 'frontend', 'err' => 'frontend_failed'];
}

/**
 * استخراج template های API از HTML و script bundle ها.
 * @return string[]
 */
function discoverApiTemplatesFromHtmlAndScripts(string $html, string $base): array {
    $base = rtrim($base, '/');
    $out = [];

    // 1) در خود HTML دنبال token= و /api/ می‌گردیم
    if (preg_match_all('~(?:https?:)?//[^\s"\']+token=[^\s"\']*~i', $html, $m1)) {
        foreach ($m1[0] as $u) {
            $out[] = normalizeDiscoveredUrl($u, $base);
        }
    }
    if (preg_match_all('~\/[A-Za-z0-9_\-\/\.]*token=~i', $html, $m2)) {
        foreach ($m2[0] as $u) $out[] = normalizeDiscoveredUrl($u, $base);
    }
    if (preg_match_all('~\/(?:api|v1|v2)\/[A-Za-z0-9_\-\/\.]{2,120}~i', $html, $m3)) {
        foreach ($m3[0] as $p) {
            // فقط مسیرهای مرتبط
            if (preg_match('/(usage|traffic|quota|sub|subscription|userinfo|service|client|stat)/i', $p)) {
                $out[] = $base . $p;
            }
        }
    }

    // 2) script src ها
    $scripts = [];
    if (preg_match_all('~<script[^>]+src=["\']([^"\']+)["\']~i', $html, $ms)) {
        foreach ($ms[1] as $src) {
            $u = normalizeDiscoveredUrl($src, $base);
            if ($u) $scripts[] = $u;
        }
    }

    // همچنین modulepreload
    if (preg_match_all('~<link[^>]+rel=["\']modulepreload["\'][^>]+href=["\']([^"\']+)["\']~i', $html, $ml)) {
        foreach ($ml[1] as $src) {
            $u = normalizeDiscoveredUrl($src, $base);
            if ($u) $scripts[] = $u;
        }
    }

    $scripts = array_values(array_unique($scripts));

    // 3) دانلود چند bundle (محدود) و اسکن رشته‌ها
    $count = 0;
    foreach ($scripts as $surl) {
        if ($count >= 4) break; // جلوگیری از سنگین شدن
        $count++;
        $res = httpGetWithHeaders($surl, 25);
        if (empty($res['ok'])) continue;
        $js = (string)($res['body'] ?? '');
        if ($js === '') continue;
        // محدود کردن برای رم
        if (strlen($js) > 1200000) {
            $js = substr($js, 0, 1200000);
        }

        // URL هایی که token= دارند (با یا بدون مقدار)
        if (preg_match_all('~["\']((?:https?:)?//[^"\']+token=)(?:\{token\}|\$\{[^\}]+\}|[^"\'&]*)["\']~i', $js, $mj1)) {
            foreach ($mj1[1] as $pref) {
                $out[] = normalizeDiscoveredUrl($pref . '{token}', $base);
            }
        }
        if (preg_match_all('~["\'](\/[A-Za-z0-9_\-\/\.]*token=)["\']~i', $js, $mj2)) {
            foreach ($mj2[1] as $pref) {
                $out[] = normalizeDiscoveredUrl($pref . '{token}', $base);
            }
        }

        // مسیرهای api که احتمالاً با token کار می‌کنند
        if (preg_match_all('~["\'](\/(?:api|v1|v2)\/[A-Za-z0-9_\-\/\.]{2,160})["\']~i', $js, $mj3)) {
            foreach ($mj3[1] as $p) {
                if (preg_match('/(usage|traffic|quota|sub|subscription|userinfo|service|client|stat)/i', $p)) {
                    $out[] = $base . $p;
                }
            }
        }
    }

    $out = array_values(array_unique(array_filter($out)));

    // فیلتر: فقط همان هاست
    $host = strtolower((string)parse_url($base, PHP_URL_HOST));
    $out2 = [];
    foreach ($out as $u) {
        $h = strtolower((string)parse_url($u, PHP_URL_HOST));
        if ($h === '' || $h === $host) $out2[] = $u;
    }
    return array_values(array_unique($out2));
}

function normalizeDiscoveredUrl(string $u, string $base): string {
    $u = trim($u);
    if ($u === '') return '';
    if (strpos($u, '//') === 0) {
        $scheme = parse_url($base, PHP_URL_SCHEME) ?: 'https';
        return $scheme . ':' . $u;
    }
    if (preg_match('~^https?://~i', $u)) return $u;
    if ($u[0] === '/') return rtrim($base, '/') . $u;
    // relative path
    return rtrim($base, '/') . '/' . ltrim($u, '/');
}

/**
 * پارس usage از headers/body (برای استفاده در کشف API)
 */
function parseUsageFromHeadersBody(array $headers, string $body, string $urlForRaw = ''): array {
    // subscription-userinfo
    $userinfo = null;
    foreach (['subscription-userinfo', 'Subscription-Userinfo'] as $k) {
        if (isset($headers[strtolower($k)])) { $userinfo = $headers[strtolower($k)]; break; }
        if (isset($headers[$k])) { $userinfo = $headers[$k]; break; }
    }
    if ($userinfo !== null && trim((string)$userinfo) !== '') {
        $p = parseSubscriptionUserinfo((string)$userinfo);
        $upload = (int)($p['upload'] ?? 0);
        $download = (int)($p['download'] ?? 0);
        $total = (int)($p['total'] ?? 0);
        $expire = normalizeExpireUnixTs($p['expire'] ?? ($p['expires'] ?? null));
        return [
            'ok' => true,
            'used_bytes' => max(0, $upload + $download),
            'total_bytes' => max(0, $total),
            'expire_ts' => ($expire && $expire > 0) ? $expire : null,
            'source' => 'subscription-userinfo',
            'raw' => ['url' => $urlForRaw],
        ];
    }

    $btrim = trim($body);
    if ($btrim !== '' && (($btrim[0] ?? '') === '{' || ($btrim[0] ?? '') === '[')) {
        $js = json_decode($btrim, true);
        if (is_array($js)) {
            $usedBytes = null; $totalBytes = null; $expireTs = null;
            foreach (['used_bytes','used','usage','traffic_used'] as $k) {
                if (isset($js[$k]) && is_numeric($js[$k])) { $usedBytes = (int)$js[$k]; break; }
            }
            foreach (['total_bytes','total','quota','traffic_total'] as $k) {
                if (isset($js[$k]) && is_numeric($js[$k])) { $totalBytes = (int)$js[$k]; break; }
            }
            foreach (['expire','expire_ts','expires_at','expires','expired_at','expireAt'] as $k) {
                    if (isset($js[$k])) {
                        $rawExp = $js[$k];
                        if (is_numeric($rawExp) || is_string($rawExp)) {
                            $tmpTs = normalizeExpireUnixTs($rawExp);
                            if ($tmpTs) { $expireTs = (int)$tmpTs; break; }
                        }
                    }
                }
            if ($usedBytes !== null && $totalBytes !== null) {
                return [
                    'ok' => true,
                    'used_bytes' => max(0, (int)$usedBytes),
                    'total_bytes' => max(0, (int)$totalBytes),
                    'expire_ts' => ($expireTs && $expireTs > 0) ? (int)$expireTs : null,
                    'source' => 'json',
                    'raw' => ['url' => $urlForRaw],
                ];
            }
        }
    }

    // آخرین تلاش: پارس متن
    $stripped = trim(preg_replace('/\s+/', ' ', strip_tags($body)));
    $norm = normalizeFaDigits($stripped);
    $unitMul = ['B'=>1,'KB'=>1024,'MB'=>1048576,'GB'=>1073741824];
    if (preg_match('/([0-9]+(?:\.[0-9]+)?)\s*(GB|MB|KB|B)\s*\/\s*([0-9]+(?:\.[0-9]+)?)\s*(GB|MB|KB|B)/i', $norm, $m)) {
        $usedBytes = (int)round(((float)$m[1]) * ($unitMul[strtoupper($m[2])] ?? 1));
        $totalBytes = (int)round(((float)$m[3]) * ($unitMul[strtoupper($m[4])] ?? 1));
        if ($totalBytes > 0) {
            return ['ok'=>true,'used_bytes'=>max(0,$usedBytes),'total_bytes'=>max(0,$totalBytes),'expire_ts'=>null,'source'=>'text','raw'=>['url'=>$urlForRaw]];
        }
    }

    return ['ok'=>false,'used_bytes'=>0,'total_bytes'=>0,'expire_ts'=>null,'source'=>'parse','err'=>'parse_fail'];
}



function computeDaysLeftFromExpire(?int $expireTs): ?int {
    if (!$expireTs || $expireTs <= 0) return null;
    $diff = $expireTs - time();
    return (int)ceil($diff / 86400);
}

function getConfigUsageCache(int $configId): ?array {
    ensureConfigUsageCacheTable();
    try {
        $pdo = db();
        $st = $pdo->prepare("SELECT * FROM config_usage_cache WHERE config_id=? LIMIT 1");
        $st->execute([(int)$configId]);
        $row = $st->fetch();
        return $row ?: null;
    } catch (Throwable $e) {
        log_error('getConfigUsageCache failed', ['msg' => $e->getMessage()]);
        return null;
    }
}

function upsertConfigUsageCache(int $configId, array $data): void {
    ensureConfigUsageCacheTable();
    $pdo = db();
    $used = (int)($data['used_bytes'] ?? 0);
    $total = (int)($data['total_bytes'] ?? 0);
    $expire = $data['expire_ts'] ?? null;
    $now = time();
    $percent = 0;
    if ($total > 0) {
        $percent = (int)floor(min(100, max(0, ($used / $total) * 100)));
    }
    $days = null;
    if (is_array($data) && array_key_exists('days_left', $data) && $data['days_left'] !== null && $data['days_left'] !== '') {
        $days = (int)$data['days_left'];
    } else {
        $days = computeDaysLeftFromExpire(is_null($expire) ? null : (int)$expire);
    }
    $meta = $data;
    unset($meta['used_bytes'], $meta['total_bytes'], $meta['expire_ts']);
    $metaJson = json_encode($meta, JSON_UNESCAPED_UNICODE);

    $sql = "INSERT INTO config_usage_cache (config_id, used_bytes, total_bytes, expire_ts, percent_used, days_left, last_checked, meta)\n"
        . "VALUES (:cid,:u,:t,:e,:p,:d,:lc,:m)\n"
        . "ON DUPLICATE KEY UPDATE\n"
        . " used_bytes=VALUES(used_bytes),\n"
        . " total_bytes=VALUES(total_bytes),\n"
        . " expire_ts=VALUES(expire_ts),\n"
        . " percent_used=VALUES(percent_used),\n"
        . " days_left=VALUES(days_left),\n"
        . " last_checked=VALUES(last_checked),\n"
        . " meta=VALUES(meta)";
    $st = $pdo->prepare($sql);
    $st->execute([
        ':cid' => $configId,
        ':u' => $used,
        ':t' => $total,
        ':e' => $expire,
        ':p' => $percent,
        ':d' => $days,
        ':lc' => $now,
        ':m' => $metaJson,
    ]);
}

/**
 * ساخت متن مصرف/انقضا برای نمایش در «سرویس‌های من»
 */
function buildUsageLineForConfig(array $cfg, ?array $cache): string {
    $created = (int)($cfg['created_at'] ?? 0);
    $duration = (int)($cfg['duration_days'] ?? 0);
    $fallbackExpire = ($created > 0 && $duration > 0) ? ($created + ($duration * 86400)) : null;

    $used = (int)($cache['used_bytes'] ?? 0);
    $total = (int)($cache['total_bytes'] ?? 0);
    $percent = (int)($cache['percent_used'] ?? 0);
    $expireTs = $cache['expire_ts'] ?? null;
    if (!$expireTs && $fallbackExpire) $expireTs = $fallbackExpire;
    $daysLeft = null;
    if (is_array($cache) && array_key_exists('days_left', $cache) && $cache['days_left'] !== null && $cache['days_left'] !== '') {
        $daysLeft = (int)$cache['days_left'];
    } else {
        $daysLeft = computeDaysLeftFromExpire($expireTs ? (int)$expireTs : null);
    }

    $lines = '';
    if ($total > 0) {
        $lines .= "میزان مصرف: <b>" . htmlspecialchars(formatGb($used)) . " از " . htmlspecialchars(formatGb($total)) . " ({$percent}%)</b>
";
    } else {
        $lines .= "میزان مصرف: <b>در حال بروزرسانی…</b>
";
    }

    // وضعیت حجم/زمان
    $volumeExpired = ($total > 0 && $used >= $total);
    $timeExpired = false;
    if ($expireTs && (int)$expireTs > 0) {
        $timeExpired = ((int)$expireTs <= time());
    } elseif ($daysLeft !== null) {
        $timeExpired = ((int)$daysLeft < 0);
    }

    if (!$volumeExpired && !$timeExpired) {
        $status = '✅ فعال';
    } else {
        $status = '⛔️ منقضی';
    }
    $lines .= "وضعیت: <b>{$status}</b>
";

    if ($daysLeft !== null) {
        $dl = (int)$daysLeft;
        if ($dl > 0) {
            $lines .= "🗓️ روزهای باقی مانده: <b>{$dl} روز</b>
";
        } elseif ($dl === 0) {
            $lines .= "🗓️ روزهای باقی مانده: <b>امروز آخرین روز</b>
";
        } else {
            $lines .= "🗓️ روزهای باقی مانده: <b>منقضی شده</b>
";
        }
    } else {
        $lines .= "🗓️ روزهای باقی مانده: <b>نامشخص</b>
";
    }

    // پیام‌های دقیق پایان
    if ($volumeExpired) {
        $lines .= "🔻 به دلیل اتمام حجم
";
        $lines .= "📦 حجم به اتمام رسیده
";
    }
    if ($timeExpired) {
        $lines .= "🔻 به دلیل اتمام زمان
";
        $lines .= "⏳ تعداد روز به اتمام رسید
";
    }

    if (!empty($cache['last_checked'])) {
        $lines .= "آخرین بروزرسانی: <b>" . htmlspecialchars(date('H:i', (int)$cache['last_checked'])) . "</b>
";
    }

    return $lines;
}

/**
 * اضافه کردن وضعیت به متن پیام، با رعایت محدودیت 4096 کاراکتر تلگرام
 */
function appendStatusToMessageText($baseText, $statusBlock) {
    $baseText = (string)$baseText;
    $statusBlock = (string)$statusBlock;

    if (mb_strlen($baseText . $statusBlock) <= 4096) {
        return $baseText . $statusBlock;
    }

    $allow = 4096 - mb_strlen($statusBlock) - 3;
    if ($allow < 0) $allow = 0;
    $trimmed = mb_substr($baseText, 0, $allow) . '...';
    return $trimmed . $statusBlock;
}

function answerCallback($cb_id, $text = '') {
    return tg('answerCallbackQuery', [
        'callback_query_id' => $cb_id,
        'text' => $text,
        'show_alert' => false,
    ]);
}

/**
 * سازگاری: بعضی بخش‌ها از answerCallbackQuery استفاده می‌کنند.
 * این تابع را اضافه می‌کنیم تا خطای Undefined Function رخ ندهد.
 */
function answerCallbackQuery($cb_id, $text = '', $show_alert = false) {
    return tg('answerCallbackQuery', [
        'callback_query_id' => $cb_id,
        'text' => (string)$text,
        'show_alert' => (bool)$show_alert,
    ]);
}

function copyMessage($to, $from, $message_id, $reply_markup = null) {
    $p = ['chat_id' => $to, 'from_chat_id' => $from, 'message_id' => $message_id];
    if ($reply_markup) $p['reply_markup'] = $reply_markup;
    return tg('copyMessage', $p);
}

function getChatMember($chat, $user_id) {
    return tg('getChatMember', ['chat_id' => $chat, 'user_id' => $user_id]);
}

function isAdmin($chat_id) {
    return (int)$chat_id === (int)ADMIN_ID;
}

/**
 * Bot Menu Button (دکمه آبی Menu)
 * - عنوان: Menu (پیش‌فرض تلگرام)
 * - فقط دستور /start نمایش داده شود
 */
function ensureBotMenuButton() {
    // فقط یک دستور
    tg('setMyCommands', [
        'commands' => [
            ['command' => 'start', 'description' => 'Menu'],
        ],
    ]);

    // منوی آبی از نوع commands
    tg('setChatMenuButton', [
        // حذف دکمه آبی (بازگشت به حالت پیش‌فرض)
        'menu_button' => ['type' => 'default'],
    ]);
}

/**
 * حذف دکمه منوی سفارشی برای یک چت (مثلاً دکمه «پنل ادمین» روی نوار پیام)
 * - فقط برای اطمینان از اینکه هیچ دکمه سفارشی باقی نماند
 */
function removeCustomChatMenuButtonForChat(int $chat_id): void {
    tg('setChatMenuButton', [
        'chat_id' => (int)$chat_id,
        'menu_button' => ['type' => 'default'],
    ]);
}


/**
 * منوی کاربران (Reply Keyboard)
 */
function userMenuKeyboard() {
    $rows = [
        // مرتب‌سازی در دو ستون + جابجایی جای دکمه‌ها طبق درخواست (تمدید چپ، خرید راست)
        [['text' => '🔄 تمدید کانفیگ'], ['text' => '🛒 خرید کانفیگ']],
        [['text' => '📂 سرویس‌های من'], ['text' => '📡 نحوه اتصال']],
        [['text' => '🆘 پشتیبانی'], ['text' => '💰 کیف پول']],
    ];

    // ردیف آخر: زیرمجموعه‌گیری + (در صورت فعال بودن) کانفیگ رایگان در دو ستون
    if (freeConfigIsEnabled()) {
        $rows[] = [
            ['text' => '👥 زیرمجموعه‌گیری'],
            ['text' => '🎁 کانفیگ رایگان'],
        ];
    } else {
        $rows[] = [
            ['text' => '👥 زیرمجموعه‌گیری'],
        ];
    }

    return [
        'keyboard' => $rows,
        'resize_keyboard' => true,
    ];
}


/**
 * منوی ادمین (Reply Keyboard)
 */
function adminMenuKeyboard() {
    return [
        'keyboard' => [
            [['text' => '🧾 سفارشات جدید'], ['text' => '♻️ درخواست‌های تمدید']],
            [['text' => '🎫 تیکت‌ها'], ['text' => '🧩 ارسال دستی کانفیگ']],
            [['text' => 'ارسال پیام'], ['text' => '📊 آمار']],
            [['text' => 'مدیریت کاربران'], ['text' => '📣 اسپانسر']],
            [['text' => '🎟 مدیریت کد تخفیف'], ['text' => '💰 شارژ کیف پول']],
            [['text' => '🗂 مدیریت کانفیگ‌ها'], ['text' => '🎁 کانفیگ رایگان']],
            [['text' => '💳 تنظیم شماره کارت']],
            [['text' => '🔙 بازگشت']],
        ],
        'resize_keyboard' => true,
    ];
}


/**
 * زیرمنوی مدیریت کاربران (Reply Keyboard)
 */
function adminUsersManageMenuKeyboard(): array {
    return [
        'keyboard' => [
            [['text' => 'کاربران'], ['text' => 'مسدود/فعال کردن']],
            [['text' => '🔙 بازگشت']],
        ],
        'resize_keyboard' => true,
    ];
}

/**
 * کیبورد عملیات کاربران مسدود (Reply Keyboard)
 */
function adminUsersBlockActionsKeyboard(): array {
    return [
        'keyboard' => [
            [['text' => '✅ فعالسازی'], ['text' => '⛔️ مسدود سازی']],
            [['text' => '🔙 بازگشت']],
        ],
        'resize_keyboard' => true,
    ];
}




/**
 * زیرمنوی ارسال پیام برای ادمین
 */
function adminMessageMenuKeyboard(): array {
    return [
        'keyboard' => [
            [['text' => 'پیام همگانی'], ['text' => 'پیام خصوصی']],
        ],
        'resize_keyboard' => true,
    ];
}


/**
 * زیرمنوی شارژ کیف پول برای ادمین
 */
function adminWalletTopupMenuKeyboard() {
    return [
        'keyboard' => [
            [['text' => '👤 شارژ کاربر تکی'], ['text' => '🌍 شارژ همگانی']],
            [['text' => '🔙 بازگشت']],
        ],
        'resize_keyboard' => true,
    ];
}



/**
 * 🗂 مدیریت کانفیگ‌ها (گروه‌ها و پلن‌ها)
 */
function adminConfigManageKeyboard(): array {
    return [
        'keyboard' => [
            [['text' => '➕ اضافه کردن'], ['text' => '✏️ ویرایش']],
            [['text' => '⛔️ خاموش/روشن گروه'], ['text' => '🗑 حذف']],
            [['text' => '🔙 بازگشت']],
        ],
        'resize_keyboard' => true,
    ];
}

/**
 * 🗑 حذف (مدیریت کانفیگ‌ها)
 */
function adminConfigDeleteKeyboard(): array {
    return [
        'keyboard' => [
            [['text' => '🗂 گروه‌ها'], ['text' => '🧾 پلن‌ها']],
            [['text' => '💳 تنظیم شماره کارت']],
            [['text' => '🔙 بازگشت']],
        ],
        'resize_keyboard' => true,
    ];
}

/**
 * مدیریت کد تخفیف (پنل ادمین)
 */
function adminDiscountMenuKeyboard() {
    return [
        'keyboard' => [
            [['text' => '➕ افزودن کد تخفیف'], ['text' => '⏳ انقضای کد تخفیف']],
            [['text' => '👤 کد تخفیف اختصاصی'], ['text' => '🌍 کد تخفیف همگانی']],
            [['text' => '🎯 گروه خاص'], ['text' => '📋 نمایش کدهای تخفیف']],
            [['text' => '🗑 حذف کد تخفیف']],
            [['text' => '🔙 بازگشت']],
        ],
        'resize_keyboard' => true,
    ];
}



/**
 * کیبورد تایید بروزرسانی کد تکراری
 */
function adminDiscountUpsertConfirmKeyboard(): array {
    return [
        'keyboard' => [
            [['text' => '✅ بروزرسانی'], ['text' => '❌ لغو']],
            [['text' => '🔙 بازگشت']],
        ],
        'resize_keyboard' => true,
    ];
}


/**
 * لیست کدها برای انتخاب (۲ ستون)
 */
function adminDiscountCodesKeyboard(array $codes, bool $withBack = true): array {
    $rows = [];
    $row = [];
    foreach ($codes as $c) {
        $row[] = ['text' => (string)$c['code']];
        if (count($row) === 2) {
            $rows[] = $row;
            $row = [];
        }
    }
    if (!empty($row)) $rows[] = $row;
    if ($withBack) $rows[] = [['text' => '🔙 بازگشت']];
    return ['keyboard' => $rows, 'resize_keyboard' => true];

}

/**
 * لیست کدها برای انتخاب + دکمه حذف همه
 */
function adminDiscountCodesKeyboardWithDeleteAll(array $codes): array {
    $kb = adminDiscountCodesKeyboard($codes, false);
    $kb['keyboard'][] = [['text' => '🗑 حذف همه']];
    $kb['keyboard'][] = [['text' => '🔙 بازگشت']];
    return $kb;
}

/**
 * بررسی وجود ستون‌های لازم برای پنل مدیریت کد تخفیف
 */
function discountAdminColumnsReady(): bool {
    if (!discountTablesExist()) return false;
    try {
        $pdo = db();
        $cols = [];
        $stmt = $pdo->query("SHOW COLUMNS FROM discount_codes");
        foreach ($stmt->fetchAll() as $r) $cols[] = (string)$r['Field'];
        return in_array('starts_at', $cols, true) && in_array('user_id', $cols, true);
    } catch (Throwable $e) {
        log_error('discountAdminColumnsReady failed', ['msg' => $e->getMessage()]);
        return false;
    }
}

/**
 * پارس تاریخ ساده (YYYY-MM-DD یا YYYY/MM/DD) به timestamp
 * اگر فقط تاریخ باشد و $endOfDay=true، پایان روز (23:59:59) در نظر گرفته می‌شود.
 * @return int|null
 */
/**
 * تبدیل تاریخ جلالی به میلادی (خروجی: [Y,M,D])
 * الگوریتم استاندارد JDF
 */
function jalali_to_gregorian($j_y, $j_m, $j_d) {
    $g_days_in_month = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
    $j_days_in_month = [31, 31, 31, 31, 31, 31, 30, 30, 30, 30, 30, 29];

    $jy = (int)$j_y - 979;
    $jm = (int)$j_m - 1;
    $jd = (int)$j_d - 1;

    $j_day_no = 365 * $jy + div($jy, 33) * 8 + div(($jy % 33) + 3, 4);
    for ($i = 0; $i < $jm; ++$i) $j_day_no += $j_days_in_month[$i];
    $j_day_no += $jd;

    $g_day_no = $j_day_no + 79;

    $gy = 1600 + 400 * div($g_day_no, 146097);
    $g_day_no = $g_day_no % 146097;

    $leap = true;
    if ($g_day_no >= 36525) {
        $g_day_no--;
        $gy += 100 * div($g_day_no, 36524);
        $g_day_no = $g_day_no % 36524;

        if ($g_day_no >= 365) $g_day_no++;
        else $leap = false;
    }

    $gy += 4 * div($g_day_no, 1461);
    $g_day_no %= 1461;

    if ($g_day_no >= 366) {
        $leap = false;
        $g_day_no--;
        $gy += div($g_day_no, 365);
        $g_day_no = $g_day_no % 365;
    }

    for ($i = 0; $g_day_no >= $g_days_in_month[$i] + (($i == 1 && $leap) ? 1 : 0); $i++) {
        $g_day_no -= $g_days_in_month[$i] + (($i == 1 && $leap) ? 1 : 0);
    }
    $gm = $i + 1;
    $gd = $g_day_no + 1;

    return [$gy, $gm, $gd];
}

/**
 * پارس تاریخ (میلادی یا جلالی) به timestamp
 * - اگر سال 13xx یا 14xx باشد جلالی فرض می‌شود.
 */
function parseDateToTimestamp(string $s, bool $endOfDay = false): ?int {
    $s = trim($s);
    if ($s === '') return null;

    // یکسان‌سازی جداکننده‌ها
    $s = str_replace(['٫', '–', '—'], ['.', '-', '-'], $s);
    $s = preg_replace('/\s+/', ' ', $s);

    // اگر فقط تاریخ بود
    if (preg_match('/^\d{4}[\/\-]\d{1,2}[\/\-]\d{1,2}$/', $s)) {
        $s .= $endOfDay ? ' 23:59:59' : ' 00:00:00';
    }

    // تشخیص جلالی: 13xx یا 14xx
    if (preg_match('/^((?:13|14)\d{2})[\/\-](\d{1,2})[\/\-](\d{1,2})(?:\s+(\d{1,2}):(\d{2})(?::(\d{2}))?)?$/u', $s, $m)) {
        $jy = (int)$m[1]; $jm = (int)$m[2]; $jd = (int)$m[3];
        $hh = isset($m[4]) ? (int)$m[4] : 0;
        $ii = isset($m[5]) ? (int)$m[5] : 0;
        $ss = isset($m[6]) ? (int)$m[6] : 0;

        [$gy, $gm, $gd] = jalali_to_gregorian($jy, $jm, $jd);
        $gs = sprintf('%04d-%02d-%02d %02d:%02d:%02d', $gy, $gm, $gd, $hh, $ii, $ss);
        $ts = strtotime($gs);
        return $ts !== false ? (int)$ts : null;
    }

    // حالت میلادی
    $s2 = str_replace('/', '-', $s);
    $ts = strtotime($s2);
    return $ts !== false ? (int)$ts : null;
}

function listDiscountCodesAll(): array {
    if (!discountTablesExist()) return [];
    $pdo = db();
    try {
        return $pdo->query("SELECT id, code, title, type, value, max_uses, used_count, starts_at, expires_at, active, user_id, created_at FROM discount_codes ORDER BY id DESC")->fetchAll();
    } catch (Throwable $e) {
        log_error('listDiscountCodesAll failed', ['msg' => $e->getMessage()]);
        return [];
    }
}

/**
 * دریافت یک کد تخفیف با مقدار code
 */
function getDiscountCodeRow(string $code) {
    if (!discountTablesExist()) return null;
    $pdo = db();
    $code = normalizeDiscountCode($code);
    $stmt = $pdo->prepare("SELECT * FROM discount_codes WHERE code=? LIMIT 1");
    $stmt->execute([$code]);
    return $stmt->fetch();
}




/**
 * خروجی صفحه‌بندی‌شده لیست کاربران برای پنل ادمین
 * @return array{text:string,kb:array}
 */
function adminUsersListPayload(int $page = 1, int $perPage = 20): array {
    $pdo = db();

    $total = (int)$pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();

    $pages = max(1, (int)ceil($total / max(1, $perPage)));
    if ($page < 1) $page = 1;
    if ($page > $pages) $page = $pages;

    $perPage = max(1, (int)$perPage);
    $offset = ($page - 1) * $perPage;

    $limit = (int)$perPage;
    $off = (int)$offset;

    $hasFreeUsage = freeUsageTableExists();

    $selectFree = $hasFreeUsage
        ? "COALESCE((SELECT used_count FROM free_config_usage f WHERE f.user_id=u.id),0) AS free_used_count,
" .
          "COALESCE((SELECT last_used_at FROM free_config_usage f WHERE f.user_id=u.id),0) AS free_last_used_at
"
        : "0 AS free_used_count,
0 AS free_last_used_at
";

    $sql =
        "SELECT u.id, u.chat_id, u.username, u.first_name, u.last_name, u.is_blocked, u.created_at,
" .
        "COALESCE((SELECT COUNT(*) FROM orders o WHERE o.user_id=u.id AND o.status='completed'),0) AS paid_orders,
" .
        "COALESCE((SELECT COUNT(*) FROM configs c WHERE c.user_id=u.id AND c.price_toman>0),0) AS paid_configs,
" .
        $selectFree .
        "FROM users u
" .
        "ORDER BY u.id DESC
" .
        "LIMIT {$limit} OFFSET {$off}";

    $rows = $pdo->query($sql)->fetchAll();

    // خلاصه بالا (اختیاری)
    $summary = '';
    try {
        $buyers = (int)$pdo->query(
            "SELECT COUNT(*) FROM users u
" .
            "WHERE EXISTS (SELECT 1 FROM orders o WHERE o.user_id=u.id AND o.status='completed')
" .
            "   OR EXISTS (SELECT 1 FROM configs c WHERE c.user_id=u.id AND c.price_toman>0)"
        )->fetchColumn();

        $freeOnly = 0;
        if ($hasFreeUsage) {
            $freeOnly = (int)$pdo->query(
                "SELECT COUNT(*) FROM users u
" .
                "WHERE NOT (
" .
                "   EXISTS (SELECT 1 FROM orders o WHERE o.user_id=u.id AND o.status='completed')
" .
                "   OR EXISTS (SELECT 1 FROM configs c WHERE c.user_id=u.id AND c.price_toman>0)
" .
                ")
" .
                "AND EXISTS (SELECT 1 FROM free_config_usage f WHERE f.user_id=u.id AND f.used_count>0)"
            )->fetchColumn();
        }

        $nonBuyers = max(0, $total - $buyers);
        $summary = "💳 خریدار: <b>{$buyers}</b> | 🎁 فقط رایگان: <b>{$freeOnly}</b> | 👤 بدون خرید: <b>{$nonBuyers}</b>
";
    } catch (Throwable $e) {
        log_error('adminUsersListPayload summary failed', ['msg' => $e->getMessage()]);
        $summary = '';
    }

    $out = "👥 <b>لیست کاربران</b>
";
    $out .= "صفحه <b>{$page}</b> از <b>{$pages}</b> | کل کاربران: <b>{$total}</b>
";
    if ($summary !== '') $out .= $summary;
    $out .= "
";

    if (!$rows) {
        $out .= "— کاربری یافت نشد —";
    } else {
        $i = 0;
        $n = count($rows);
        foreach ($rows as $r) {
            $i++;
	            $cid = (int)($r['chat_id'] ?? 0);
	            $hasUsername = !empty($r['username']);
	            $uname = $hasUsername ? ('@' . htmlspecialchars((string)$r['username'])) : '—';
            $fullName = trim((string)($r['first_name'] ?? '') . ' ' . (string)($r['last_name'] ?? ''));
            if ($fullName === '') $fullName = '—';
            $status = ((int)($r['is_blocked'] ?? 0) === 1) ? '⛔️ بلاک' : '✅ فعال';
            $created = jdate_from_timestamp((int)($r['created_at'] ?? 0));

	            // 👤 لینک پروفایل فقط برای کاربرانی که یوزرنیم ندارند
	            // نکته: در خیلی از نسخه‌ها، tg://user?id=... بهترین سازگاری را دارد.
	            if ($cid > 0 && !$hasUsername) {
	                $profileLink = '<a href="tg://user?id=' . $cid . '">👤 پروفایل</a>';
	                $nameHtml = ($fullName !== '—')
	                    ? ('<a href="tg://user?id=' . $cid . '">' . htmlspecialchars($fullName) . '</a>')
	                    : htmlspecialchars($fullName);
	            } else {
	                $profileLink = '';
	                $nameHtml = htmlspecialchars($fullName);
	            }

            $paidOrders = (int)($r['paid_orders'] ?? 0);
            $paidConfigs = (int)($r['paid_configs'] ?? 0);
            $freeUsed = (int)($r['free_used_count'] ?? 0);
            $freeLast = (int)($r['free_last_used_at'] ?? 0);

            $isBuyer = ($paidOrders > 0 || $paidConfigs > 0);
            $isFreeOnly = (!$isBuyer && $freeUsed > 0);

            if ($isBuyer) {
                $badge = "💳 <b>خریدار</b>";
            } elseif ($isFreeOnly) {
                $badge = "🎁 <b>فقط رایگان</b>";
            } else {
                $badge = "👤 <b>بدون خرید</b>";
            }

	            $line = "• #" . (int)$r['id'] . " | <code>" . htmlspecialchars((string)$r['chat_id']) . "</code> | {$uname}";
	            if ($profileLink !== '') $line .= " | {$profileLink}";
	            $line .= " | {$badge}
";
	            $out .= $line;
	            $out .= "  نام: <b>{$nameHtml}</b> | {$status}
";
            $out .= "  خرید موفق: <b>{$paidOrders}</b> | کانفیگ پولی: <b>{$paidConfigs}</b>";
            if ($hasFreeUsage) {
                $out .= " | رایگان: <b>{$freeUsed}</b>";
                if ($freeLast > 0) {
                    $out .= " (آخرین: <b>" . jdate_from_timestamp($freeLast) . "</b>)";
                }
            }
            $out .= "
";
            $out .= "  ثبت: <b>{$created}</b>\n\n";
            if ($i < $n) {
                $out .= "────────────────────────────\n\n";
            }

        }
    }

    $kbRows = [];

    // صفحه‌بندی
    $prevCb = ($page > 1) ? ('admin:users:list:' . ($page - 1)) : 'noop';
    $nextCb = ($page < $pages) ? ('admin:users:list:' . ($page + 1)) : 'noop';

    $kbRows[] = [
        ['text' => '⬅️ قبلی', 'callback_data' => $prevCb],
        ['text' => "{$page}/{$pages}", 'callback_data' => 'noop'],
        ['text' => 'بعدی ➡️', 'callback_data' => $nextCb],
    ];
    $kbRows[] = [
        ['text' => '🔄 بروزرسانی', 'callback_data' => 'admin:users:list:' . $page],
    ];

    $kb = ['inline_keyboard' => $kbRows];

    return ['text' => $out, 'kb' => $kb];
}


/**
 * خروجی لیست کاربران مسدود برای پنل ادمین
 */
function adminBlockedUsersListText(int $limit = 100): string {
    $pdo = db();
    $limit = max(1, min(500, (int)$limit));

    $total = 0;
    try {
        $total = (int)$pdo->query("SELECT COUNT(*) FROM users WHERE is_blocked=1")->fetchColumn();
    } catch (Throwable $e) {
        log_error('adminBlockedUsersListText count failed', ['msg' => $e->getMessage()]);
    }

    $stmt = $pdo->prepare("SELECT id, chat_id, username, first_name, last_name, created_at FROM users WHERE is_blocked=1 ORDER BY id DESC LIMIT {$limit}");
    $stmt->execute();
    $rows = $stmt->fetchAll();

    $out = "⛔️ <b>کاربران مسدود</b>\n";
    $out .= "کل کاربران مسدود: <b>{$total}</b>\n\n";

    if (!$rows) {
        $out .= "— هیچ کاربر مسدودی وجود ندارد —";
        return $out;
    }

    $i = 0;
    $n = count($rows);
    foreach ($rows as $r) {
        $i++;
        $uname = !empty($r['username']) ? ('@' . htmlspecialchars((string)$r['username'])) : '—';
        $fullName = trim((string)($r['first_name'] ?? '') . ' ' . (string)($r['last_name'] ?? ''));
        if ($fullName === '') $fullName = '—';
        $created = jdate_from_timestamp((int)($r['created_at'] ?? 0));

        $out .= "• #" . (int)$r['id'] . " | <code>" . htmlspecialchars((string)$r['chat_id']) . "</code> | {$uname}\n";
        $out .= "  نام: <b>" . htmlspecialchars($fullName) . "</b>\n";
        $out .= "  ثبت: <b>{$created}</b>\n\n";
        if ($i < $n) {
            $out .= "────────────────────────────\n\n";
        }
    }

    return $out;
}


function startKeyboardFor($chat_id) {
    return isAdmin($chat_id) ? adminMenuKeyboard() : userMenuKeyboard();
}

/**
 * ==============================
 * تنظیمات ربات (Bot Settings)
 * ==============================
 * جدول: settings (k,v,updated_at)
 */
function botSettingsTableExists(): bool {
    static $exists = null;
    if ($exists !== null) return (bool)$exists;
    try {
        $pdo = db();
        $stmt = $pdo->query("SHOW TABLES LIKE 'settings'");
        $exists = ($stmt && $stmt->fetchColumn()) ? true : false;
    } catch (Throwable $e) {
        log_error('settings table check failed', ['msg' => $e->getMessage()]);
        $exists = false;
    }
    return (bool)$exists;
}

function getBotSetting(string $key, $default = null) {
    if (!isset($GLOBALS['__bot_settings_cache']) || !is_array($GLOBALS['__bot_settings_cache'])) {
        $GLOBALS['__bot_settings_cache'] = [];
    }
    $cache =& $GLOBALS['__bot_settings_cache'];
    if (array_key_exists($key, $cache)) return $cache[$key];

    if (!botSettingsTableExists()) {
        $cache[$key] = $default;
        return $default;
    }

    try {
        $pdo = db();
        $stmt = $pdo->prepare("SELECT v FROM settings WHERE k=? LIMIT 1");
        $stmt->execute([$key]);
        $v = $stmt->fetchColumn();
        if ($v === false || $v === null) {
            $cache[$key] = $default;
            return $default;
        }
        $cache[$key] = $v;
        return $v;
    } catch (Throwable $e) {
        log_error('getBotSetting failed', ['key' => $key, 'msg' => $e->getMessage()]);
        $cache[$key] = $default;
        return $default;
    }
}

function setBotSetting(string $key, $value): bool {
    if (!botSettingsTableExists()) return false;
    try {
        $pdo = db();
        $now = time();
        $stmt = $pdo->prepare(
            "INSERT INTO settings (k, v, updated_at) VALUES (?, ?, ?)
" .
            "ON DUPLICATE KEY UPDATE v=VALUES(v), updated_at=VALUES(updated_at)"
        );
        $stmt->execute([$key, (string)$value, $now]);

        // بروزرسانی کش
        if (!isset($GLOBALS['__bot_settings_cache']) || !is_array($GLOBALS['__bot_settings_cache'])) {
            $GLOBALS['__bot_settings_cache'] = [];
        }
        $GLOBALS['__bot_settings_cache'][$key] = (string)$value;

        return true;
    } catch (Throwable $e) {
        log_error('setBotSetting failed', ['key' => $key, 'msg' => $e->getMessage()]);
        return false;
    }
}


/**
 * ==============================
 * تنظیم شماره کارت (قابل مدیریت از داخل ربات)
 * ==============================
 * ذخیره در settings:
 * - card_number
 * - card_holder
 */
function ensureCardSettingsMigrated(): void {
    if (!botSettingsTableExists()) return;

    $num = getBotSetting('card_number', '');
    $holder = getBotSetting('card_holder', '');

    $num = is_string($num) ? trim($num) : '';
    $holder = is_string($holder) ? trim($holder) : '';

    if ($num !== '' && $holder !== '') return;

    $c = require __DIR__ . '/config.php';
    $cfgNum = is_string($c['CARD_NUMBER'] ?? null) ? trim((string)$c['CARD_NUMBER']) : '';
    $cfgHolder = is_string($c['CARD_HOLDER'] ?? null) ? trim((string)$c['CARD_HOLDER']) : '';

    if ($num === '' && $cfgNum !== '') {
        setBotSetting('card_number', $cfgNum);
    }
    if ($holder === '' && $cfgHolder !== '') {
        setBotSetting('card_holder', $cfgHolder);
    }
}

function getCardNumberValue(): string {
    ensureCardSettingsMigrated();
    $v = getBotSetting('card_number', '');
    $v = is_string($v) ? trim($v) : '';
    if ($v !== '') return $v;

    // fallback (فقط برای مواقعی که settings موجود نیست)
    $c = require __DIR__ . '/config.php';
    return is_string($c['CARD_NUMBER'] ?? null) ? trim((string)$c['CARD_NUMBER']) : '';
}

function getCardHolderValue(): string {
    ensureCardSettingsMigrated();
    $v = getBotSetting('card_holder', '');
    $v = is_string($v) ? trim($v) : '';
    if ($v !== '') return $v;

    $c = require __DIR__ . '/config.php';
    return is_string($c['CARD_HOLDER'] ?? null) ? trim((string)$c['CARD_HOLDER']) : '';
}


/**
 * ==============================
 * کانفیگ رایگان
 * ==============================
 */
function freeConfigIsEnabled(): bool {
    $v = getBotSetting('free_config_enabled', '0');
    return ((string)$v === '1' || (string)$v === 'true');
}

function getFreeSubLink(): string {
    $v = getBotSetting('free_sub_link', '');
    return is_string($v) ? trim($v) : '';
}

function adminFreeConfigPanelPayload(): array {
    $enabled = freeConfigIsEnabled();
    $link = getFreeSubLink();

    $status = $enabled ? '✅ فعال' : '⛔️ غیرفعال';
    $linkStatus = ($link !== '') ? '✅ تنظیم شده' : '⚠️ تنظیم نشده';

    $txt = "🎁 <b>کانفیگ رایگان</b>

";
    $txt .= "وضعیت دکمه برای کاربران: <b>{$status}</b>
";
    $txt .= "وضعیت لینک ساب: <b>{$linkStatus}</b>
";
    if ($link !== '') {
        $txt .= "
لینک فعلی:
<code>" . htmlspecialchars($link) . "</code>
";
    }

    $kb = [
        'inline_keyboard' => [
            [
                ['text' => ($enabled ? '⛔️ غیرفعال کردن' : '✅ فعال کردن'), 'callback_data' => 'admin:freecfg:toggle'],
            ],
            [
                ['text' => '🔗 تنظیم لینک', 'callback_data' => 'admin:freecfg:setlink'],
                ['text' => '🔄 بروزرسانی', 'callback_data' => 'admin:freecfg:refresh'],
            ],
        ],
    ];

    return ['text' => $txt, 'kb' => $kb];
}

/**
 * ==============================
 * ثبت استفاده از کانفیگ رایگان
 * ==============================
 */
function freeUsageTableExists(): bool {
    static $exists = null;
    if ($exists !== null) return (bool)$exists;
    try {
        $pdo = db();
        $stmt = $pdo->query("SHOW TABLES LIKE 'free_config_usage'");
        $exists = ($stmt && $stmt->fetchColumn()) ? true : false;
    } catch (Throwable $e) {
        log_error('free_config_usage table check failed', ['msg' => $e->getMessage()]);
        $exists = false;
    }
    return (bool)$exists;
}

function recordFreeConfigUsageForChatId(int $chat_id): void {
    if (!freeUsageTableExists()) return;
    try {
        $pdo = db();
        $u = getUserByChatId($chat_id);
        if (!$u) return;

        $now = time();
        $stmt = $pdo->prepare(
            "INSERT INTO free_config_usage (user_id, used_count, last_used_at) VALUES (?, 1, ?)
" .
            "ON DUPLICATE KEY UPDATE used_count = used_count + 1, last_used_at = VALUES(last_used_at)"
        );
        $stmt->execute([(int)$u['id'], $now]);
    } catch (Throwable $e) {
        log_error('recordFreeConfigUsageForChatId failed', ['msg' => $e->getMessage()]);
    }
}


/**
 * ==============================
 * اسپانسر (کانال اضافی برای جوین اجباری)
 * ==============================
 * در جدول settings با کلید sponsor_channel ذخیره می‌شود.
 */
function normalizeChannelIdentifier(string $input): string {
    $s = trim($input);
    if ($s === '') return '';

    // URL های t.me
    if (stripos($s, 'https://t.me/') === 0) {
        $s = substr($s, strlen('https://t.me/'));
    } elseif (stripos($s, 'http://t.me/') === 0) {
        $s = substr($s, strlen('http://t.me/'));
    } elseif (stripos($s, 't.me/') === 0) {
        $s = substr($s, strlen('t.me/'));
    }

    // حذف پارامترها
    $s = preg_replace('~\?.*$~u', '', $s);
    $s = trim($s, "/ \t\n\r\0\x0B");

    // لینک‌های خصوصی (t.me/+...) یا joinchat قابل چک نیستند
    if ($s !== '' && ($s[0] === '+' || stripos($s, 'joinchat') === 0)) {
        return '';
    }

    // شناسه عددی (مثل -100...)
    if (preg_match('/^-?\d+$/', $s)) {
        return $s;
    }

    // @username یا username
    if ($s[0] === '@') {
        $s = substr($s, 1);
    }
    if (preg_match('/^[A-Za-z0-9_]{5,32}$/', $s)) {
        return '@' . $s;
    }

    return '';
}


/**
 * ==============================
 * جوین اجباری (قابل مدیریت از داخل ربات)
 * ==============================
 * - لیست کانال‌ها در جدول settings و در کلید join_channels به صورت JSON ذخیره می‌شود.
 * - هر آیتم: {channel: string, url: string|null, button_text: string}
 * - برای سازگاری: اگر join_channels خالی باشد، از config.php + کلید قدیمی sponsor_channel مهاجرت می‌کند.
 */
function joinChannelsSettingKey(): string {
    return 'join_channels';
}

function joinChannelsMigratedKey(): string {
    return 'join_channels_migrated';
}

/**
 * مهاجرت اولیه از config.php (REQUIRED_CHANNELS) و کلید قدیمی sponsor_channel به join_channels
 * - فقط یکبار انجام می‌شود (با فلگ join_channels_migrated)
 * - بعد از مهاجرت، دیگر از config.php کانالی دوباره اضافه نمی‌شود (تا حذف از پنل درست کار کند)
 */
function migrateJoinChannelsFromConfigIfNeeded(): void {
    if (!botSettingsTableExists()) return;

    $migrated = (string)getBotSetting(joinChannelsMigratedKey(), '0');
    if ($migrated === '1') return;

    $rawNow = getBotSetting(joinChannelsSettingKey(), '');
    $rawNow = is_string($rawNow) ? trim($rawNow) : '';
    if ($rawNow !== '') {
        // اگر قبلاً چیزی ذخیره شده، مهاجرت را تمام شده علامت بزن
        setBotSetting(joinChannelsMigratedKey(), '1');
        return;
    }

    // از config.php فقط برای مهاجرت اولیه می‌خوانیم (اگر وجود داشت)
    $c = @include __DIR__ . '/config.php';
    $base = [];
    if (is_array($c) && isset($c['REQUIRED_CHANNELS']) && is_array($c['REQUIRED_CHANNELS'])) {
        $base = $c['REQUIRED_CHANNELS'];
    }

    $items = [];
    $i = 1;
    foreach ($base as $raw) {
        $ch = normalizeChannelIdentifier((string)$raw);
        if ($ch === '') continue;
        $items[] = [
            'channel' => $ch,
            'url' => channelToUrl($ch),
            'button_text' => 'عضویت در کانال ' . $i,
        ];
        $i++;
    }

    // کلید قدیمی sponsor_channel (در نسخه‌های خیلی قدیمی)
    $legacySponsor = getBotSetting('sponsor_channel', '');
    if (is_string($legacySponsor) && trim($legacySponsor) !== '') {
        $ch = normalizeChannelIdentifier((string)$legacySponsor);
        if ($ch !== '') {
            $items[] = [
                'channel' => $ch,
                'url' => channelToUrl($ch),
                'button_text' => 'عضویت در کانال ' . $i,
            ];
        }
    }

    // حذف تکراری‌ها
    $out = [];
    $seen = [];
    foreach ($items as $it) {
        $ch = (string)($it['channel'] ?? '');
        if ($ch === '' || isset($seen[$ch])) continue;
        $seen[$ch] = 1;
        $out[] = $it;
    }

    if (!empty($out)) {
        setBotSetting(joinChannelsSettingKey(), json_encode($out, JSON_UNESCAPED_UNICODE));
    }

    // مهاجرت انجام شد
    setBotSetting(joinChannelsMigratedKey(), '1');

    // کلید قدیمی دیگر استفاده نمی‌شود
    if (is_string($legacySponsor) && trim($legacySponsor) !== '') {
        setBotSetting('sponsor_channel', '');
    }
}

function getJoinRequiredChannelsItems(): array {
    // اگر جدول settings نیست، از config.php می‌خوانیم (حالت قدیمی)
    if (!botSettingsTableExists()) {
        $c = require __DIR__ . '/config.php';
        $base = $c['REQUIRED_CHANNELS'] ?? [];
        if (!is_array($base)) $base = [];

        $items = [];
        $i = 1;
        foreach ($base as $raw) {
            $ch = normalizeChannelIdentifier((string)$raw);
            if ($ch === '') continue;
            $items[] = [
                'channel' => $ch,
                'url' => channelToUrl($ch),
                'button_text' => 'عضویت در کانال ' . $i,
            ];
            $i++;
        }
        return $items;
    }

    // مهاجرت اولیه (فقط یکبار)
    migrateJoinChannelsFromConfigIfNeeded();

    $raw = getBotSetting(joinChannelsSettingKey(), '');
    $raw = is_string($raw) ? trim($raw) : '';

    $items = [];
    if ($raw !== '') {
        $decoded = json_decode($raw, true);
        if (is_array($decoded)) $items = $decoded;
    }

    // sanitize
    $out = [];
    $seen = [];
    $i = 1;
    foreach ($items as $it) {
        if (!is_array($it)) continue;
        $ch = normalizeChannelIdentifier((string)($it['channel'] ?? ''));
        if ($ch === '' || isset($seen[$ch])) continue;
        $seen[$ch] = 1;

        $btn = trim((string)($it['button_text'] ?? ''));
        if ($btn === '') $btn = 'عضویت در کانال ' . $i;

        $url = $it['url'] ?? null;
        $url = is_string($url) ? trim($url) : '';
        if ($url === '') $url = channelToUrl($ch) ?: '';

        $out[] = [
            'channel' => $ch,
            'url' => $url,
            'button_text' => $btn,
        ];
        $i++;
    }

    // ذخیره نسخه تمیز (برای حذف/ویرایش پایدار)
    $encoded = json_encode($out, JSON_UNESCAPED_UNICODE);
    if ($encoded && $encoded !== $raw) {
        setBotSetting(joinChannelsSettingKey(), $encoded);
    }

    return $out;
}

function setJoinRequiredChannelsItems(array $items): bool {
    if (!botSettingsTableExists()) return false;
    $encoded = json_encode(array_values($items), JSON_UNESCAPED_UNICODE);
    if (!$encoded) return false;
    return setBotSetting(joinChannelsSettingKey(), $encoded);
}

/**
 * پنل اسپانسر (مدیریت جوین اجباری)
 */
function adminSponsorPanelPayload(): array {
    $items = getJoinRequiredChannelsItems();

    $txt = "📣 <b>اسپانسر | جوین اجباری</b>

";
    if (empty($items)) {
        $txt .= "وضعیت: <b>⛔️ تنظیم نشده</b>

";
        $txt .= "✅ از دکمه <b>تنظیم کانال</b> برای افزودن کانال اسپانسر/جوین اجباری استفاده کنید.";
    } else {
        $txt .= "وضعیت: <b>✅ فعال</b>
";
        $txt .= "کانال‌های جوین اجباری:
";
        $n = 1;
        foreach ($items as $it) {
            $ch = (string)$it['channel'];
            $btn = (string)$it['button_text'];
            $txt .= "• <b>{$n}</b>) <code>" . htmlspecialchars($ch) . "</code> — <i>" . htmlspecialchars($btn) . "</i>
";
            $n++;
        }
        $txt .= "
ℹ️ کاربران برای استفاده از ربات باید عضو همه موارد بالا باشند.";
    }

    // ترتیب دکمه‌ها برای اینکه «تنظیم کانال» در سمت راست باشد:
    $kbRows = [
        [
            ['text' => '🔄 بروزرسانی', 'callback_data' => 'admin:sponsor:refresh'],
            ['text' => '🔗 تنظیم کانال', 'callback_data' => 'admin:sponsor:add'],
        ],
    ];

    if (!empty($items)) {
        $kbRows[] = [
            ['text' => '🗑 حذف اسپانسر', 'callback_data' => 'admin:sponsor:delete_menu'],
        ];
    }

    return [
        'text' => $txt,
        'kb' => ['inline_keyboard' => $kbRows],
    ];
}
/**
 * تبدیل شناسه کانال به URL قابل کلیک
 * ورودی‌های پشتیبانی‌شده:
 * - @username
 * - https://t.me/username
 * - t.me/username
 * خروجی: string|null
 */
function channelToUrl($channel) {
    if (!is_string($channel)) return null;
    $ch = trim($channel);
    if ($ch === '') return null;

    // اگر خودش URL است
    if (stripos($ch, 'https://t.me/') === 0) return $ch;
    if (stripos($ch, 'http://t.me/') === 0) return 'https://t.me/' . substr($ch, strlen('http://t.me/'));
    if (stripos($ch, 't.me/') === 0) return 'https://' . $ch;

    // @username
    if ($ch[0] === '@') {
        $u = substr($ch, 1);
        if ($u !== '') return 'https://t.me/' . $u;
    }

    // اگر آیدی عددی کانال باشد (مثل -100...) لینک عمومی قابل ساخت نیست
    return null;
}

/**
 * ============================== 
 * اجبار عضویت در کانال‌ها (الزامی)
 * ==============================
 *
 * ✅ قبل از هر پردازش (پیام/دستور/کال‌بک) باید فراخوانی شود.
 * ✅ هر بار وضعیت عضویت به صورت زنده با getChatMember بررسی می‌شود.
 * ✅ هیچ وضعیت «عضو بودن قبلی» ذخیره نمی‌شود.
 * ✅ فقط status های معتبر: member | administrator | creator
 */
function forceJoinCheck($chat_id, array $context = []) {
    // کانال‌های جوین اجباری از دیتابیس (settings->join_channels)
    $items = getJoinRequiredChannelsItems();
    if (empty($items)) return true;

    // در چت خصوصی، chat_id همان user_id است؛ با این حال برای اطمینان قابل تزریق است.
    $user_id = isset($context['user_id']) ? (int)$context['user_id'] : (int)$chat_id;

    // فقط این وضعیت‌ها معتبر هستند
    $validStatuses = ['member', 'administrator', 'creator'];

    // لیست کانال‌هایی که کاربر در آن‌ها عضو نیست
    $missing = [];
    foreach ($items as $it) {
        $ch = (string)($it['channel'] ?? '');
        if ($ch === '') continue;

        $res = getChatMember($ch, $user_id);
        $st = null;
        if ($res && !empty($res['ok']) && isset($res['result']['status'])) {
            $st = (string)$res['result']['status'];
        }

        if (!in_array($st, $validStatuses, true)) {
            $missing[] = $it;
        }
    }

    if (!empty($missing)) {
        // کاربر عضو نیست => قفل
        $txt  = "⚠️ برای استفاده از خدمات ربات باید در کانال(های) زیر عضو باشید:\n\n";

        $kbRows = [];
        $idx = 1;
        foreach ($missing as $it) {
            $ch = trim((string)($it['channel'] ?? ''));
            $url = trim((string)($it['url'] ?? ''));
            $btnText = trim((string)($it['button_text'] ?? ''));

            if ($btnText === '') $btnText = 'عضویت در کانال ' . $idx;

            // لینک در متن (در صورت امکان)
            if ($url !== '') {
                $txt .= '• <a href="' . htmlspecialchars($url, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . '">' .
                    htmlspecialchars($ch, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . "</a>\n";
                $kbRows[] = [
                    ['text' => $btnText, 'url' => $url],
                ];
            } else {
                // اگر URL قابل ساخت نبود
                $txt .= '• ' . htmlspecialchars($ch, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . "\n";
            }

            $idx++;
        }

        // دکمه چک مجدد
        $kbRows[] = [
            ['text' => '✅ عضو شدم', 'callback_data' => 'force_join:check'],
        ];

        $kb = ['inline_keyboard' => $kbRows];

        // اگر از کال‌بک صدا زده شده، برای جلوگیری از لودینگ دکمه پاسخ بدهیم
        if (!empty($context['callback_query_id'])) {
            answerCallback((string)$context['callback_query_id']);
        }

        // اگر پیام فعلی خودِ پیام اجبار عضویت باشد، همان را ادیت کنیم؛ وگرنه پیام جدید بفرستیم.
        $mid = !empty($context['message_id']) ? (int)$context['message_id'] : 0;
        $mtext = $context['message_text'] ?? '';
        $isJoinPrompt = is_string($mtext) && (
            mb_strpos($mtext, 'برای استفاده از خدمات ربات') !== false ||
            mb_strpos($mtext, 'باید در کانال') !== false ||
            mb_strpos($mtext, '✅ عضو شدم') !== false
        );

        if ($mid > 0 && $isJoinPrompt) {
            $ed = editMessageText($chat_id, $mid, $txt, $kb);
            if (!$ed || empty($ed['ok'])) {
                sendMessage($chat_id, $txt, $kb);
            }
        } else {
            sendMessage($chat_id, $txt, $kb);
        }

        return false;
    }

    return true;
}



function enforceJoinIfEnabled($chat_id) {
    return forceJoinCheck((int)$chat_id, ['user_id' => (int)$chat_id]);
}

/**
 * فرمت قیمت (تومان)
 */
function formatToman($amount) {
    $n = (int)$amount;
    return number_format($n, 0, '.', ',');
}

/**
 * عنوان گروه پلن‌ها (برای نمایش در UI)
 */

/**
 * گروه‌های پلن (برای خرید و مدیریت)
 * - اگر جدول plan_groups وجود نداشته باشد، به صورت پیش‌فرض bulk/vip استفاده می‌شود.
 */
function planGroupsTableExists(): bool {
    try {
        $pdo = db();
        $stmt = $pdo->query("SHOW TABLES LIKE 'plan_groups'");
        $row = $stmt->fetch();
        return !empty($row);
    } catch (Throwable $e) {
        return false;
    }
}

function ensureDefaultPlanGroups(): void {
    if (!planGroupsTableExists()) return;
    try {
        $pdo = db();
        // bulk و vip را اگر وجود ندارند اضافه می‌کنیم
        $pdo->prepare("INSERT IGNORE INTO plan_groups (group_key, title, active) VALUES ('bulk', 'حجمی | مناسب وبگردی | مجازی', 1)")->execute();
        $pdo->prepare("INSERT IGNORE INTO plan_groups (group_key, title, active) VALUES ('vip', 'VIP | مناسب ترید | گیمینگ', 1)")->execute();
    } catch (Throwable $e) {
        // ignore
    }
}

function listPlanGroupsActive(): array {
    if (!planGroupsTableExists()) {
        return [
            ['group_key' => 'bulk', 'title' => 'حجمی | مناسب وبگردی | مجازی', 'active' => 1],
            ['group_key' => 'vip', 'title' => 'VIP | مناسب ترید | گیمینگ', 'active' => 1],
        ];
    }
    ensureDefaultPlanGroups();
    try {
        $pdo = db();
        $stmt = $pdo->query("SELECT group_key, title, active FROM plan_groups WHERE active=1 ORDER BY id ASC");
        return $stmt->fetchAll();
    } catch (Throwable $e) {
        return [
            ['group_key' => 'bulk', 'title' => 'حجمی | مناسب وبگردی | مجازی', 'active' => 1],
            ['group_key' => 'vip', 'title' => 'VIP | مناسب ترید | گیمینگ', 'active' => 1],
        ];
    }
}

/**
 * لیست همه گروه‌ها (برای پنل ادمین) – شامل فعال/غیرفعال
 */
function listPlanGroupsAll(): array {
    if (!planGroupsTableExists()) {
        return [
            ['group_key' => 'bulk', 'title' => 'حجمی | مناسب وبگردی | مجازی', 'active' => 1],
            ['group_key' => 'vip', 'title' => 'VIP | مناسب ترید | گیمینگ', 'active' => 1],
        ];
    }
    ensureDefaultPlanGroups();
    try {
        $pdo = db();
        $stmt = $pdo->query("SELECT group_key, title, active FROM plan_groups ORDER BY id ASC");
        return $stmt->fetchAll();
    } catch (Throwable $e) {
        return [
            ['group_key' => 'bulk', 'title' => 'حجمی | مناسب وبگردی | مجازی', 'active' => 1],
            ['group_key' => 'vip', 'title' => 'VIP | مناسب ترید | گیمینگ', 'active' => 1],
        ];
    }
}

/**
 * فعال/غیرفعال کردن گروه
 */
function setPlanGroupActive(string $group_key, int $active): bool {
    if (!planGroupsTableExists()) return false;
    $group_key = strtolower(trim($group_key));
    $active = ($active ? 1 : 0);
    if ($group_key === '') return false;
    try {
        $pdo = db();
        $stmt = $pdo->prepare("UPDATE plan_groups SET active=? WHERE group_key=?");
        $stmt->execute([$active, $group_key]);
        return $stmt->rowCount() >= 0;
    } catch (Throwable $e) {
        return false;
    }
}



/**
 * کیبورد انتخاب گروه برای خرید سرویس (کاربر)
 */
function buildBuyGroupKeyboard(): array {
    $groups = listPlanGroupsActive();
    $rows = [];
    $row = [];

    foreach ($groups as $g) {
        $key = (string)($g['group_key'] ?? '');
        $title = (string)($g['title'] ?? $key);
        if ($key === '') continue;

        $row[] = ['text' => $title, 'callback_data' => 'buy:group:' . $key];
        if (count($row) === 2) {
            $rows[] = $row;
            $row = [];
        }
    }
    if (!empty($row)) $rows[] = $row;

    $rows[] = [['text' => '❌ لغو', 'callback_data' => 'buy:group:cancel']];

    return ['inline_keyboard' => $rows];
}

function getPlanGroupByKey(string $group_key) {
    $group_key = strtolower(trim($group_key));
    if (!planGroupsTableExists()) {
        if ($group_key === 'vip') return ['group_key' => 'vip', 'title' => 'VIP | مناسب ترید | گیمینگ', 'active' => 1];
        return ['group_key' => 'bulk', 'title' => 'حجمی | مناسب وبگردی | مجازی', 'active' => 1];
    }
    ensureDefaultPlanGroups();
    try {
        $pdo = db();
        $stmt = $pdo->prepare("SELECT * FROM plan_groups WHERE group_key=? LIMIT 1");
        $stmt->execute([$group_key]);
        return $stmt->fetch();
    } catch (Throwable $e) {
        return null;
    }
}

function createPlanGroup(string $group_key, string $title): bool {
    if (!planGroupsTableExists()) return false;
    $group_key = strtolower(trim($group_key));
    $title = trim($title);
    if ($group_key === '' || $title === '') return false;
    try {
        $pdo = db();
        $stmt = $pdo->prepare("INSERT INTO plan_groups (group_key, title, active) VALUES (?, ?, 1)");
        return (bool)$stmt->execute([$group_key, $title]);
    } catch (Throwable $e) {
        return false;
    }
}

function updatePlanGroupTitle(string $group_key, string $title): bool {
    if (!planGroupsTableExists()) return false;
    $group_key = strtolower(trim($group_key));
    $title = trim($title);
    if ($group_key === '' || $title === '') return false;
    try {
        $pdo = db();
        $stmt = $pdo->prepare("UPDATE plan_groups SET title=? WHERE group_key=?");
        $stmt->execute([$title, $group_key]);
        return $stmt->rowCount() >= 0;
    } catch (Throwable $e) {
        return false;
    }
}

function deletePlanGroupCascade(string $group_key): array {
    $group_key = strtolower(trim($group_key));
    $out = ['ok' => false, 'plans_deleted' => 0, 'group_deleted' => 0];
    if (!planGroupsTableExists()) return $out;
    try {
        $pdo = db();
        // اول پلن‌ها
        $st1 = $pdo->prepare("DELETE FROM plans WHERE group_key=?");
        $st1->execute([$group_key]);
        $out['plans_deleted'] = (int)$st1->rowCount();

        // سپس گروه
        $st2 = $pdo->prepare("DELETE FROM plan_groups WHERE group_key=?");
        $st2->execute([$group_key]);
        $out['group_deleted'] = (int)$st2->rowCount();

        $out['ok'] = true;
        return $out;
    } catch (Throwable $e) {
        return $out;
    }
}

function plansUserCountColumnExists(): bool {
    try {
        $pdo = db();
        $stmt = $pdo->query("SHOW COLUMNS FROM plans LIKE 'user_count'");
        $row = $stmt->fetch();
        return !empty($row);
    } catch (Throwable $e) {
        return false;
    }
}

function planGroupTitle(string $key): string {
    $g = getPlanGroupByKey($key);
    if (!empty($g) && !empty($g['title'])) {
        return (string)$g['title'];
    }

    $key = strtolower(trim($key));
    if ($key === 'vip') {
        return 'VIP | مناسب ترید | گیمینگ';
    }
    return 'حجمی | مناسب وبگردی | مجازی';
}

/**
 * نمایش حجم (گیگ) برای حالت نامحدود
 */
function formatPlanDataLabel(int $data_gb): string {
    return ($data_gb <= 0) ? 'نامحدود' : ((int)$data_gb . ' گیگ');
}


/**
 * ساخت متن «اطلاعات پرداخت» برای خرید
 */
function buildBuyPaymentMessage(array $data): string {
    $base = (int)($data['base_price_toman'] ?? $data['price_toman'] ?? 0);
    $discount = (int)($data['discount_amount_toman'] ?? 0);
    $final = (int)($data['final_price_toman'] ?? max(0, $base - $discount));

    $name = (string)($data['config_name'] ?? '');
    $code = (string)($data['discount_code'] ?? '');

    $groupKey = (string)($data['plan_group_key'] ?? $data['group_key'] ?? 'bulk');
    $msg = "🧾 اطلاعات پرداخت\n\n";
    $msg .= "گروه: <b>" . htmlspecialchars(planGroupTitle($groupKey)) . "</b>\n";
    $msg .= "پلن: <b>" . htmlspecialchars((string)($data['title'] ?? '—')) . "</b>\n";
    $msg .= "پروتکل: <b>" . MAIN_PROTOCOL . "</b>\n";
    $msg .= "حجم: <b>" . htmlspecialchars(formatPlanDataLabel((int)($data['data_gb'] ?? 0))) . "</b>\n";
    $msg .= "مدت: <b>" . (int)($data['duration_days'] ?? 0) . " روز</b>\n";
    $msg .= "نام سرویس: <b>" . htmlspecialchars($name) . "</b>\n\n";

    if ($discount > 0) {
        $msg .= "قیمت پایه: <b>" . formatToman($base) . " تومان</b>\n";
        $msg .= "کد تخفیف: <b>" . htmlspecialchars($code !== '' ? $code : '—') . "</b>\n";
        $msg .= "مبلغ تخفیف: <b>" . formatToman($discount) . " تومان</b>\n";
        $msg .= "مبلغ نهایی: <b>" . formatToman($final) . " تومان</b>\n";
    } else {
        $msg .= "قیمت: <b>" . formatToman($final) . " تومان</b>\n";
    }

    $msg .= "\nروش پرداخت را انتخاب کنید:";
    return $msg;
}

/**
 * ساخت متن «اطلاعات پرداخت» برای تمدید
 */
function buildRenewPaymentMessage(array $cfg, array $data): string {
    $base = (int)($data['base_price_toman'] ?? 0);
    $discount = (int)($data['discount_amount_toman'] ?? 0);
    $final = (int)($data['final_price_toman'] ?? max(0, $base - $discount));
    $code = (string)($data['discount_code'] ?? '');

    $msg = "🔁 تمدید کانفیگ\n\n";
    $msg .= "نام کانفیگ: <b>" . htmlspecialchars((string)($cfg['name'] ?? '—')) . "</b>\n";
    $msg .= "پروتکل: <b>" . htmlspecialchars((string)($cfg['protocol'] ?? MAIN_PROTOCOL)) . "</b>\n";
    $msg .= "پلن/حجم: <b>" . (int)($cfg['data_gb'] ?? 0) . " گیگ</b>\n";
    $msg .= "مدت: <b>" . (int)($cfg['duration_days'] ?? 0) . " روز</b>\n\n";

    if ($discount > 0) {
        $msg .= "قیمت پایه: <b>" . formatToman($base) . " تومان</b>\n";
        $msg .= "کد تخفیف: <b>" . htmlspecialchars($code !== '' ? $code : '—') . "</b>\n";
        $msg .= "مبلغ تخفیف: <b>" . formatToman($discount) . " تومان</b>\n";
        $msg .= "مبلغ نهایی: <b>" . formatToman($final) . " تومان</b>\n";
    } else {
        $msg .= "قیمت: <b>" . formatToman($final) . " تومان</b>\n";
    }

    $msg .= "\nروش پرداخت را انتخاب کنید:";
    return $msg;
}

/**
 * تبدیل تاریخ میلادی به جلالی و نمایش در ساعت تهران
 * ورودی: timestamp ثانیه
 */
function jdate_from_timestamp($ts) {
    $g = explode('-', date('Y-m-d', (int)$ts));
    [$jy, $jm, $jd] = gregorian_to_jalali((int)$g[0], (int)$g[1], (int)$g[2]);
    $time = date('H:i', (int)$ts);
    return sprintf('%04d/%02d/%02d %s', $jy, $jm, $jd, $time);
}

function div($a, $b) { return (int)($a / $b); }

function gregorian_to_jalali($g_y, $g_m, $g_d) {
    $g_days_in_month = [0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
    $j_days_in_month = [0, 31, 31, 31, 31, 31, 31, 30, 30, 30, 30, 30, 29];

    $gy = $g_y - 1600;
    $gm = $g_m - 1;
    $gd = $g_d - 1;

    $g_day_no = 365 * $gy + div($gy + 3, 4) - div($gy + 99, 100) + div($gy + 399, 400);
    for ($i = 0; $i < $gm; ++$i) $g_day_no += $g_days_in_month[$i + 1];
    if ($gm > 1 && (($g_y % 4 == 0 && $g_y % 100 != 0) || ($g_y % 400 == 0))) $g_day_no++;
    $g_day_no += $gd;

    $j_day_no = $g_day_no - 79;
    $j_np = div($j_day_no, 12053);
    $j_day_no %= 12053;

    $jy = 979 + 33 * $j_np + 4 * div($j_day_no, 1461);
    $j_day_no %= 1461;

    if ($j_day_no >= 366) {
        $jy += div($j_day_no - 1, 365);
        $j_day_no = ($j_day_no - 1) % 365;
    }

    for ($i = 0; $i < 11 && $j_day_no >= $j_days_in_month[$i + 1]; ++$i) {
        $j_day_no -= $j_days_in_month[$i + 1];
    }

    $jm = $i + 1;
    $jd = $j_day_no + 1;

    return [$jy, $jm, $jd];
}

/**
 * ثبت/آپدیت کاربر
 */
function ensureUser($message) {
    $chat_id = (int)$message['chat']['id'];
    $username = $message['from']['username'] ?? null;
    $first = $message['from']['first_name'] ?? '';
    $last = $message['from']['last_name'] ?? '';

    $pdo = db();
    $now = time();

    // تشخیص کاربر جدید قبل از upsert
    $existsStmt = $pdo->prepare("SELECT id FROM users WHERE chat_id=? LIMIT 1");
    $existsStmt->execute([$chat_id]);
    $existingId = (int)$existsStmt->fetchColumn();
    $is_new = $existingId ? false : true;

    $stmt = $pdo->prepare(
        "INSERT INTO users (chat_id, username, first_name, last_name, created_at) VALUES (?, ?, ?, ?, ?)\n" .
        "ON DUPLICATE KEY UPDATE username=VALUES(username), first_name=VALUES(first_name), last_name=VALUES(last_name)"
    );
    $stmt->execute([$chat_id, $username, $first, $last, $now]);

    $id = $pdo->lastInsertId();
    if ($id) {
        ensureWalletRow((int)$id);
        return ['id' => (int)$id, 'is_new' => true];
    }

    $uid = $existingId;
    if (!$uid) {
        $stmt2 = $pdo->prepare("SELECT id FROM users WHERE chat_id=? LIMIT 1");
        $stmt2->execute([$chat_id]);
        $uid = (int)$stmt2->fetchColumn();
    }
    if ($uid) ensureWalletRow($uid);
    return ['id' => (int)$uid, 'is_new' => $is_new];
}

function ensureUserByChatId($chat_id) {
    $pdo = db();
    $stmt = $pdo->prepare("SELECT id FROM users WHERE chat_id=? LIMIT 1");
    $stmt->execute([(int)$chat_id]);
    $id = $stmt->fetchColumn();
    if ($id) return (int)$id;

    $now = time();
    $stmt2 = $pdo->prepare("INSERT INTO users (chat_id, created_at) VALUES (?, ?)");
    $stmt2->execute([(int)$chat_id, $now]);
    $uid = (int)$pdo->lastInsertId();
    if ($uid) ensureWalletRow($uid);
    return $uid;
}

function getUserByChatId($chat_id) {
    $pdo = db();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE chat_id = ? LIMIT 1");
    $stmt->execute([(int)$chat_id]);
    return $stmt->fetch();
}


/**
 * ==============================
 * کیف پول (Wallet)
 * ==============================
 * جدول‌ها:
 * - wallets(user_id,balance_toman,updated_at)
 * - wallet_transactions(...)
 */
function walletsTableExists(): bool {
    static $exists = null;
    if ($exists !== null) return (bool)$exists;
    try {
        $pdo = db();
        $stmt = $pdo->query("SHOW TABLES LIKE 'wallets'");
        $exists = ($stmt && $stmt->fetchColumn()) ? true : false;
    } catch (Throwable $e) {
        log_error('wallets table check failed', ['msg' => $e->getMessage()]);
        $exists = false;
    }
    return (bool)$exists;
}

function walletTxTableExists(): bool {
    static $exists = null;
    if ($exists !== null) return (bool)$exists;
    try {
        $pdo = db();
        $stmt = $pdo->query("SHOW TABLES LIKE 'wallet_transactions'");
        $exists = ($stmt && $stmt->fetchColumn()) ? true : false;
    } catch (Throwable $e) {
        log_error('wallet_transactions table check failed', ['msg' => $e->getMessage()]);
        $exists = false;
    }
    return (bool)$exists;
}

function walletTopupsTableExists(): bool {
    static $exists = null;
    if ($exists !== null) return (bool)$exists;
    try {
        $pdo = db();
        $stmt = $pdo->query("SHOW TABLES LIKE 'wallet_topups'");
        $exists = ($stmt && $stmt->fetchColumn()) ? true : false;
    } catch (Throwable $e) {
        log_error('wallet_topups table check failed', ['msg' => $e->getMessage()]);
        $exists = false;
    }
    return (bool)$exists;
}

function walletBulkCreditsTableExists(): bool {
    static $exists = null;
    if ($exists !== null) return (bool)$exists;
    try {
        $pdo = db();
        $stmt = $pdo->query("SHOW TABLES LIKE 'wallet_bulk_credits'");
        $exists = ($stmt && $stmt->fetchColumn()) ? true : false;
    } catch (Throwable $e) {
        log_error('wallet_bulk_credits table check failed', ['msg' => $e->getMessage()]);
        $exists = false;
    }
    return (bool)$exists;
}

function ensureWalletRow(int $user_id): void {
    if (!walletsTableExists()) return;
    try {
        $pdo = db();
        $now = time();
        $stmt = $pdo->prepare("INSERT IGNORE INTO wallets (user_id, balance_toman, updated_at) VALUES (?, 0, ?)");
        $stmt->execute([$user_id, $now]);
    } catch (Throwable $e) {
        log_error('ensureWalletRow failed', ['user_id' => $user_id, 'msg' => $e->getMessage()]);
    }
}

function getWalletBalanceByUserId(int $user_id): int {
    if (!walletsTableExists()) return 0;
    try {
        ensureWalletRow($user_id);
        $pdo = db();
        $stmt = $pdo->prepare("SELECT balance_toman FROM wallets WHERE user_id=? LIMIT 1");
        $stmt->execute([$user_id]);
        $b = $stmt->fetchColumn();
        return $b === false || $b === null ? 0 : (int)$b;
    } catch (Throwable $e) {
        log_error('getWalletBalanceByUserId failed', ['user_id' => $user_id, 'msg' => $e->getMessage()]);
        return 0;
    }
}

function getWalletBalanceByChatId(int $chat_id): int {
    $u = getUserByChatId($chat_id);
    if (!$u) return 0;
    return getWalletBalanceByUserId((int)$u['id']);
}

/**
 * برداشت از کیف پول
 * @return array{ok:bool, tx_id:int|null, balance:int, error?:string}
 */
function walletDebit(int $user_id, int $amount_toman, ?int $order_id = null, ?int $renewal_id = null, array $meta = []): array {
    if ($amount_toman <= 0) return ['ok' => false, 'tx_id' => null, 'balance' => getWalletBalanceByUserId($user_id), 'error' => 'amount_invalid'];
    if (!walletsTableExists() || !walletTxTableExists()) return ['ok' => false, 'tx_id' => null, 'balance' => getWalletBalanceByUserId($user_id), 'error' => 'wallet_tables_missing'];

    $pdo = db();
    $now = time();
    try {
        ensureWalletRow($user_id);
        $pdo->beginTransaction();

        $q = $pdo->prepare("SELECT balance_toman FROM wallets WHERE user_id=? FOR UPDATE");
        $q->execute([$user_id]);
        $bal = (int)$q->fetchColumn();

        if ($bal < $amount_toman) {
            $pdo->rollBack();
            return ['ok' => false, 'tx_id' => null, 'balance' => $bal, 'error' => 'insufficient'];
        }

        $newBal = $bal - $amount_toman;

        $pdo->prepare("UPDATE wallets SET balance_toman=?, updated_at=? WHERE user_id=?")
            ->execute([$newBal, $now, $user_id]);

        $metaJson = !empty($meta) ? json_encode($meta, JSON_UNESCAPED_UNICODE) : null;
        $stmt = $pdo->prepare("INSERT INTO wallet_transactions (user_id, type, amount_toman, order_id, renewal_id, created_at, meta) VALUES (?,?,?,?,?,?,?)");
        $stmt->execute([$user_id, 'debit', $amount_toman, $order_id, $renewal_id, $now, $metaJson]);
        $txId = (int)$pdo->lastInsertId();

        $pdo->commit();
        return ['ok' => true, 'tx_id' => $txId, 'balance' => $newBal];
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        log_error('walletDebit failed', ['user_id' => $user_id, 'amount' => $amount_toman, 'msg' => $e->getMessage()]);
        return ['ok' => false, 'tx_id' => null, 'balance' => getWalletBalanceByUserId($user_id), 'error' => 'db_error'];
    }
}

/**
 * افزایش موجودی کیف پول
 * @return array{ok:bool, tx_id:int|null, balance:int}
 */
function walletCredit(int $user_id, int $amount_toman, ?int $order_id = null, ?int $renewal_id = null, array $meta = []): array {
    if ($amount_toman <= 0) return ['ok' => false, 'tx_id' => null, 'balance' => getWalletBalanceByUserId($user_id)];
    if (!walletsTableExists() || !walletTxTableExists()) return ['ok' => false, 'tx_id' => null, 'balance' => getWalletBalanceByUserId($user_id)];

    $pdo = db();
    $now = time();
    try {
        ensureWalletRow($user_id);
        $pdo->beginTransaction();

        $q = $pdo->prepare("SELECT balance_toman FROM wallets WHERE user_id=? FOR UPDATE");
        $q->execute([$user_id]);
        $bal = (int)$q->fetchColumn();

        $newBal = $bal + $amount_toman;
        $pdo->prepare("UPDATE wallets SET balance_toman=?, updated_at=? WHERE user_id=?")
            ->execute([$newBal, $now, $user_id]);

        $metaJson = !empty($meta) ? json_encode($meta, JSON_UNESCAPED_UNICODE) : null;
        $stmt = $pdo->prepare("INSERT INTO wallet_transactions (user_id, type, amount_toman, order_id, renewal_id, created_at, meta) VALUES (?,?,?,?,?,?,?)");
        $stmt->execute([$user_id, 'credit', $amount_toman, $order_id, $renewal_id, $now, $metaJson]);
        $txId = (int)$pdo->lastInsertId();

        $pdo->commit();
        return ['ok' => true, 'tx_id' => $txId, 'balance' => $newBal];
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        log_error('walletCredit failed', ['user_id' => $user_id, 'amount' => $amount_toman, 'msg' => $e->getMessage()]);
        return ['ok' => false, 'tx_id' => null, 'balance' => getWalletBalanceByUserId($user_id)];
    }
}

function walletRefund(int $user_id, int $amount_toman, ?int $order_id = null, ?int $renewal_id = null, array $meta = []): array {
    // همان credit ولی نوع refund
    if ($amount_toman <= 0) return ['ok' => false, 'tx_id' => null, 'balance' => getWalletBalanceByUserId($user_id)];
    if (!walletsTableExists() || !walletTxTableExists()) return ['ok' => false, 'tx_id' => null, 'balance' => getWalletBalanceByUserId($user_id)];

    $pdo = db();
    $now = time();
    try {
        ensureWalletRow($user_id);
        $pdo->beginTransaction();

        $q = $pdo->prepare("SELECT balance_toman FROM wallets WHERE user_id=? FOR UPDATE");
        $q->execute([$user_id]);
        $bal = (int)$q->fetchColumn();

        $newBal = $bal + $amount_toman;
        $pdo->prepare("UPDATE wallets SET balance_toman=?, updated_at=? WHERE user_id=?")
            ->execute([$newBal, $now, $user_id]);

        $metaJson = !empty($meta) ? json_encode($meta, JSON_UNESCAPED_UNICODE) : null;
        $stmt = $pdo->prepare("INSERT INTO wallet_transactions (user_id, type, amount_toman, order_id, renewal_id, created_at, meta) VALUES (?,?,?,?,?,?,?)");
        $stmt->execute([$user_id, 'refund', $amount_toman, $order_id, $renewal_id, $now, $metaJson]);
        $txId = (int)$pdo->lastInsertId();

        $pdo->commit();
        return ['ok' => true, 'tx_id' => $txId, 'balance' => $newBal];
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        log_error('walletRefund failed', ['user_id' => $user_id, 'amount' => $amount_toman, 'msg' => $e->getMessage()]);
        return ['ok' => false, 'tx_id' => null, 'balance' => getWalletBalanceByUserId($user_id)];
    }
}


/**
 * ==============================
 * کد تخفیف (Discount Codes)
 * ==============================
 * جدول‌ها:
 * - discount_codes
 * - discount_usages
 */
function discountTablesExist(): bool {
    static $exists = null;
    if ($exists !== null) return (bool)$exists;
    try {
        $pdo = db();
        $stmt = $pdo->query("SHOW TABLES LIKE 'discount_codes'");
        $exists = ($stmt && $stmt->fetchColumn()) ? true : false;
        if ($exists) {
            $stmt2 = $pdo->query("SHOW TABLES LIKE 'discount_usages'");
            $exists = ($stmt2 && $stmt2->fetchColumn()) ? true : false;
        }
    } catch (Throwable $e) {
        log_error('discount tables check failed', ['msg' => $e->getMessage()]);
        $exists = false;
    }
    return (bool)$exists;
}

function normalizeDiscountCode(string $code): string {
    $code = trim($code);
    $code = str_replace([' ', '　', "
", "
", "	"], '', $code);
    return mb_strtoupper($code);
}

/**
 * فقط اعتبارسنجی و محاسبه تخفیف (بدون مصرف/ثبت)
 * @return array{ok:bool, code_id:int|null, discount:int, final:int, msg:string}
 */
function previewDiscount(string $code, int $base_price_toman, int $user_id = 0): array {
    if (!discountTablesExist()) return ['ok' => false, 'code_id' => null, 'discount' => 0, 'final' => $base_price_toman, 'msg' => '❌ سیستم کد تخفیف فعال نیست.'];

    $code = normalizeDiscountCode($code);
    if ($code === '') return ['ok' => false, 'code_id' => null, 'discount' => 0, 'final' => $base_price_toman, 'msg' => '❌ کد تخفیف خالی است.'];

    $pdo = db();
    $now = time();

    $stmt = $pdo->prepare("SELECT * FROM discount_codes WHERE code=? LIMIT 1");
    $stmt->execute([$code]);
    $row = $stmt->fetch();
    if (!$row) return ['ok' => false, 'code_id' => null, 'discount' => 0, 'final' => $base_price_toman, 'msg' => '❌ کد تخفیف نامعتبر است.'];

    if ((int)$row['active'] !== 1) return ['ok' => false, 'code_id' => (int)$row['id'], 'discount' => 0, 'final' => $base_price_toman, 'msg' => '❌ این کد تخفیف غیرفعال است.'];
    if (!empty($row['expires_at']) && (int)$row['expires_at'] > 0 && (int)$row['expires_at'] < $now) {
        return ['ok' => false, 'code_id' => (int)$row['id'], 'discount' => 0, 'final' => $base_price_toman, 'msg' => '❌ این کد تخفیف منقضی شده است.'];
    }



    $startsAt = (int)($row['starts_at'] ?? 0);
    if ($startsAt > 0 && $startsAt > $now) {
        return ['ok' => false, 'code_id' => (int)$row['id'], 'discount' => 0, 'final' => $base_price_toman, 'msg' => '❌ این کد تخفیف هنوز فعال نشده است.'];
    }

    $onlyUser = (int)($row['user_id'] ?? 0);
    if ($onlyUser > 0 && $user_id > 0 && $onlyUser !== $user_id) {
        return ['ok' => false, 'code_id' => (int)$row['id'], 'discount' => 0, 'final' => $base_price_toman, 'msg' => '❌ این کد تخفیف مخصوص کاربر دیگری است.'];
    }
    if ($onlyUser > 0 && $user_id <= 0) {
        return ['ok' => false, 'code_id' => (int)$row['id'], 'discount' => 0, 'final' => $base_price_toman, 'msg' => '❌ این کد تخفیف مخصوص یک کاربر است.'];
    }

    $maxUses = (int)($row['max_uses'] ?? 0);
    $used = (int)($row['used_count'] ?? 0);
    if ($maxUses > 0 && $used >= $maxUses) {
        return ['ok' => false, 'code_id' => (int)$row['id'], 'discount' => 0, 'final' => $base_price_toman, 'msg' => '❌ ظرفیت استفاده از این کد تمام شده است.'];
    }

    $type = (string)$row['type'];
    $val = (int)$row['value'];
    $discount = 0;
    if ($base_price_toman <= 0) $discount = 0;
    elseif ($type === 'percent') {
        $discount = (int)floor($base_price_toman * $val / 100);
    } else { // fixed
        $discount = $val;
    }
    if ($discount < 0) $discount = 0;
    if ($discount > $base_price_toman) $discount = $base_price_toman;

    $final = max(0, $base_price_toman - $discount);

    return ['ok' => true, 'code_id' => (int)$row['id'], 'discount' => $discount, 'final' => $final, 'msg' => '✅ کد تخفیف اعمال شد.'];
}



/**
 * ==============================
 * تخفیف گروه خاص (Group-Specific Discount)
 * ==============================
 * نگهداری در settings با کلید: discount_group_map
 * فرمت JSON: {"group_key": discount_code_id, ...}
 */
function getDiscountGroupMap(): array {
    $raw = getBotSetting('discount_group_map', '{}');
    if (!is_string($raw) || trim($raw) === '') $raw = '{}';
    $j = json_decode($raw, true);
    if (!is_array($j)) return [];
    $out = [];
    foreach ($j as $k => $v) {
        $gk = strtolower(trim((string)$k));
        if ($gk === '') continue;
        $cid = (int)$v;
        if ($cid <= 0) continue;
        $out[$gk] = $cid;
    }
    return $out;
}

function setDiscountGroupMap(array $map): bool {
    // نرمال‌سازی
    $out = [];
    foreach ($map as $k => $v) {
        $gk = strtolower(trim((string)$k));
        $cid = (int)$v;
        if ($gk === '' || $cid <= 0) continue;
        $out[$gk] = $cid;
    }
    return setBotSetting('discount_group_map', json_encode($out, JSON_UNESCAPED_UNICODE));
}

function assignDiscountCodeToGroup(string $group_key, int $code_id): bool {
    $group_key = strtolower(trim($group_key));
    if ($group_key === '' || $code_id <= 0) return false;

    $map = getDiscountGroupMap();

    // هر کد فقط برای یک گروه (برای جلوگیری از تداخل)
    foreach ($map as $gk => $cid) {
        if ((int)$cid === (int)$code_id && $gk !== $group_key) {
            unset($map[$gk]);
        }
    }

    $map[$group_key] = (int)$code_id;
    return setDiscountGroupMap($map);
}

function removeDiscountFromGroup(string $group_key): bool {
    $group_key = strtolower(trim($group_key));
    if ($group_key === '') return false;
    $map = getDiscountGroupMap();
    if (!isset($map[$group_key])) return true;
    unset($map[$group_key]);
    return setDiscountGroupMap($map);
}

function getRestrictedGroupForDiscountCodeId(int $code_id): ?string {
    if ($code_id <= 0) return null;
    $map = getDiscountGroupMap();
    foreach ($map as $gk => $cid) {
        if ((int)$cid === (int)$code_id) return (string)$gk;
    }
    return null;
}

/**
 * previewDiscount + اعمال محدودیت گروه
 */
function previewDiscountForGroup(string $code, int $base_price_toman, int $user_id, string $plan_group_key): array {
    $prev = previewDiscount($code, $base_price_toman, $user_id);
    if (empty($prev['ok'])) return $prev;

    $cid = (int)($prev['code_id'] ?? 0);
    $plan_group_key = strtolower(trim($plan_group_key));

    $restrictedTo = getRestrictedGroupForDiscountCodeId($cid);
    if ($restrictedTo !== null && $restrictedTo !== '' && $plan_group_key !== '' && $restrictedTo !== $plan_group_key) {
        $g = getPlanGroupByKey($restrictedTo);
        $gTitle = is_array($g) && !empty($g['title']) ? (string)$g['title'] : $restrictedTo;
        return [
            'ok' => false,
            'msg' => "❌ این کد تخفیف فقط برای گروه «" . $gTitle . "» قابل استفاده است.",
        ];
    }

    return $prev;
}
/**
 * مصرف/ثبت کد تخفیف هنگام ایجاد سفارش/تمدید
 */
function consumeDiscount(int $code_id, int $user_id, int $discount_amount_toman, ?int $order_id = null, ?int $renewal_id = null): bool {
    if (!discountTablesExist()) return false;
    if ($code_id < 1 || $discount_amount_toman < 0) return false;

    $pdo = db();
    $now = time();

    try {
        $pdo->beginTransaction();

        $q = $pdo->prepare("SELECT * FROM discount_codes WHERE id=? FOR UPDATE");
        $q->execute([$code_id]);
        $row = $q->fetch();
        if (!$row || (int)$row['active'] !== 1) {
            $pdo->rollBack();
            return false;
        }

        if (!empty($row['expires_at']) && (int)$row['expires_at'] > 0 && (int)$row['expires_at'] < $now) {
            $pdo->rollBack();
            return false;
        }

        $startsAt = (int)($row['starts_at'] ?? 0);
        if ($startsAt > 0 && $startsAt > $now) {
            $pdo->rollBack();
            return false;
        }

        $onlyUser = (int)($row['user_id'] ?? 0);
        if ($onlyUser > 0 && $onlyUser !== (int)$user_id) {
            $pdo->rollBack();
            return false;
        }


        $maxUses = (int)($row['max_uses'] ?? 0);
        $used = (int)($row['used_count'] ?? 0);
        if ($maxUses > 0 && $used >= $maxUses) {
            $pdo->rollBack();
            return false;
        }

        $pdo->prepare("UPDATE discount_codes SET used_count = used_count + 1 WHERE id=?")->execute([$code_id]);

        $stmt = $pdo->prepare("INSERT INTO discount_usages (code_id, user_id, order_id, renewal_id, discount_amount_toman, used_at) VALUES (?,?,?,?,?,?)");
        $stmt->execute([$code_id, $user_id, $order_id, $renewal_id, $discount_amount_toman, $now]);

        $pdo->commit();
        return true;
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        log_error('consumeDiscount failed', ['code_id' => $code_id, 'msg' => $e->getMessage()]);
        return false;
    }
}

/**
 * State Machine
 */
function setState($chat_id, $state, $data = null, $expires_at_ts = null) {
    $pdo = db();
    $j = $data !== null ? json_encode($data, JSON_UNESCAPED_UNICODE) : null;
    $now = time();

    $stmt = $pdo->prepare(
        "INSERT INTO states (chat_id, state, data, expires_at, updated_at) VALUES (?, ?, ?, ?, ?)\n" .
        "ON DUPLICATE KEY UPDATE state=VALUES(state), data=VALUES(data), expires_at=VALUES(expires_at), updated_at=VALUES(updated_at)"
    );
    $stmt->execute([(int)$chat_id, $state, $j, $expires_at_ts, $now]);
}

function getState($chat_id) {
    $pdo = db();
    $stmt = $pdo->prepare("SELECT * FROM states WHERE chat_id=? LIMIT 1");
    $stmt->execute([(int)$chat_id]);
    $row = $stmt->fetch();
    if (!$row) return null;

    if (!empty($row['data'])) {
        $row['data'] = json_decode($row['data'], true);
        if (!is_array($row['data'])) $row['data'] = [];
    } else {
        $row['data'] = [];
    }

    // بررسی انقضاء
    if (!empty($row['expires_at']) && (int)$row['expires_at'] > 0 && (int)$row['expires_at'] < time()) {
        // برای اینکه پیام لغو را هم بتوانیم ارسال کنیم، وضعیت را با فلگ expired برمی‌گردانیم
        $row['expired'] = true;
        clearState($chat_id);
        return $row;
    }

    $row['expired'] = false;
    return $row;
}

function clearState($chat_id) {
    $pdo = db();
    $pdo->prepare("DELETE FROM states WHERE chat_id=?")->execute([(int)$chat_id]);
}

/**
 * پلن‌ها
 */
function listActivePlans() {
    $pdo = db();
    $stmt = $pdo->query("SELECT * FROM plans WHERE active=1 ORDER BY price_toman ASC");
    return $stmt->fetchAll();
}

function listActivePlansByGroup(string $group_key) {
    $pdo = db();
    $stmt = $pdo->prepare("SELECT * FROM plans WHERE active=1 AND group_key=? ORDER BY price_toman ASC");
    $stmt->execute([trim($group_key)]);
    return $stmt->fetchAll();
}

function getPlan($id) {
    $pdo = db();
    $stmt = $pdo->prepare("SELECT * FROM plans WHERE id=? LIMIT 1");
    $stmt->execute([(int)$id]);
    return $stmt->fetch();
}

/**
 * افزودن کانفیگ برای کاربر
 */
function addConfigForUser($user_id, $name, $protocol, $plan_id, $link, $data_gb, $duration_days, $price_toman) {
    $pdo = db();
    $now = time();
    $stmt = $pdo->prepare(
        "INSERT INTO configs (user_id, name, protocol, plan_id, link, data_gb, duration_days, price_toman, created_at, active)\n" .
        "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 1)"
    );
    $stmt->execute([
        (int)$user_id,
        $name,
        $protocol,
        $plan_id !== null ? (int)$plan_id : null,
        $link,
        (int)$data_gb,
        (int)$duration_days,
        (int)$price_toman,
        $now,
    ]);
    return (int)$pdo->lastInsertId();
}

function listUserConfigs($chat_id) {
    $pdo = db();
    $user = getUserByChatId($chat_id);
    if (!$user) return [];
    $stmt = $pdo->prepare("SELECT * FROM configs WHERE user_id=? AND active=1 ORDER BY id DESC");
    $stmt->execute([(int)$user['id']]);
    return $stmt->fetchAll();
}


/**
 * دریافت کانفیگ یک کاربر (بر اساس chat_id) با کنترل مالکیت
 */
function getUserConfigById($chat_id, int $config_id, bool $activeOnly = true) {
    $pdo = db();
    $user = getUserByChatId($chat_id);
    if (!$user) return null;

    $sql = "SELECT * FROM configs WHERE id=? AND user_id=? ";
    if ($activeOnly) $sql .= "AND active=1 ";
    $sql .= "LIMIT 1";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([(int)$config_id, (int)$user['id']]);
    $cfg = $stmt->fetch();
    return $cfg ?: null;
}

/**
 * تغییر نام کانفیگ (فقط برای مالک)
 */
function renameUserConfig($chat_id, int $config_id, string $newName): bool {
    $pdo = db();
    $user = getUserByChatId($chat_id);
    if (!$user) return false;

    $stmt = $pdo->prepare("UPDATE configs SET name=? WHERE id=? AND user_id=? AND active=1");
    $stmt->execute([$newName, (int)$config_id, (int)$user['id']]);
    return $stmt->rowCount() > 0;
}

/**
 * حذف نرم (غیرفعال کردن) کانفیگ (فقط برای مالک)
 */
function deactivateUserConfig($chat_id, int $config_id): bool {
    $pdo = db();
    $user = getUserByChatId($chat_id);
    if (!$user) return false;

    $stmt = $pdo->prepare("UPDATE configs SET active=0 WHERE id=? AND user_id=? AND active=1");
    $stmt->execute([(int)$config_id, (int)$user['id']]);
    return $stmt->rowCount() > 0;
}


/**
 * حذف کامل کانفیگ از دیتابیس (Hard Delete) – فقط برای مالک
 */
function deleteUserConfigHard($chat_id, int $config_id): bool {
    $pdo = db();
    $user = getUserByChatId($chat_id);
    if (!$user) return false;

    try {
        // حذف کش مصرف
        ensureConfigUsageCacheTable();
        $st1 = $pdo->prepare("DELETE FROM config_usage_cache WHERE config_id=?");
        $st1->execute([(int)$config_id]);
    } catch (Throwable $e) {
        // ignore
    }

    try {
        $stmt = $pdo->prepare("DELETE FROM configs WHERE id=? AND user_id=?");
        $stmt->execute([(int)$config_id, (int)$user['id']]);
        return $stmt->rowCount() > 0;
    } catch (Throwable $e) {
        return false;
    }
}


/**
 * نمایش «سرویس‌های من» + دکمه‌های مدیریت (تغییر نام / حذف)
 * - این تابع، پیام لیست سرویس‌ها را (با کیبورد اصلی کاربر) ارسال می‌کند
 * - سپس یک پیام جداگانه با دکمه‌های اینلاین مدیریت می‌فرستد (تا کیبورد اصلی حفظ شود)
 */

/**
 * تولید لینک QR Code (توسط سرویس عمومی) - تلگرام خودش تصویر را دانلود می‌کند
 * مزیت: نیاز به کتابخانه/افزونه روی سرور نیست.
 */
function makeQrUrl(string $text, int $size = 420): string {
    $text = trim($text);
    if ($text === '') return '';
    $s = max(150, min(1000, (int)$size));
    return 'https://api.qrserver.com/v1/create-qr-code/?size=' . $s . 'x' . $s . '&data=' . rawurlencode($text);
}

/**
 * ساخت متن جزئیات یک سرویس برای نمایش به کاربر (همراه با مصرف/انقضا)
 */
function buildServiceDetailsText(array $cfg, ?array $cache = null): string {
    $msg  = "📄 <b>جزئیات سرویس</b>

";
    $msg .= "📛 نام سرویس: <b>" . htmlspecialchars($cfg['name']) . "</b>
";
    $msg .= "📦 حجم: <b>" . htmlspecialchars(formatPlanDataLabel((int)$cfg['data_gb'])) . "</b>
";
    $msg .= buildUsageLineForConfig($cfg, $cache);
    $msg .= "⏳ مدت: <b>" . (int)$cfg['duration_days'] . " روز</b>
";
    $msg .= "🗓 تاریخ ساخت: <b>" . jdate_from_timestamp((int)$cfg['created_at']) . "</b>
";
    $msg .= "🔗 لینک ساب:
<code>" . htmlspecialchars($cfg['link']) . "</code>

";
    $msg .= "📱 <b>راهنما:</b> QR را اسکن کنید یا لینک را کپی کنید.";
    return $msg;
}




function showUserServices($chat_id) {
    $cfgs = listUserConfigs($chat_id);
    if (!$cfgs) {
        sendMessage($chat_id, "📂 هنوز سرویسی برای شما فعال نیست.", userMenuKeyboard());
        return;
    }

    $rows = [];
    foreach ($cfgs as $c) {
        $shortName = mb_strimwidth((string)$c['name'], 0, 28, '…', 'UTF-8');
        $rows[] = [[
            'text' => "📌 " . $shortName,
            'callback_data' => 'cfg:view:' . (int)$c['id'],
        ]];
    }

    $kb = ['inline_keyboard' => $rows];

    $msg = "📂 <b>سرویس‌های من</b>

";
    $msg .= "👇 روی سرویس موردنظر بزنید تا <b>QR Code</b> و اطلاعات نمایش داده شود.";
    sendMessage($chat_id, $msg, $kb);
}






function formatUserLine($u) {
    $first = trim(($u['first_name'] ?? '') . ' ' . ($u['last_name'] ?? ''));
    if ($first === '') $first = '—';
    $parts = [];
    $parts[] = "👤 <b>{$first}</b>";
    if (!empty($u['username'])) $parts[] = '@' . $u['username'];
    $parts[] = '🆔 ' . ($u['chat_id'] ?? '—');
    return implode(' | ', $parts);
}

/**
 * پروتکل‌ها (برای سازگاری) – در این نسخه فقط V2RAY
 */
function getProtocols() {
    return [MAIN_PROTOCOL];
}



// ================== Referral (زیرمجموعه‌گیری) ==================

/**
 * دریافت username ربات (برای لینک دعوت) با کش فایل
 */
function getBotUsernameCached() {
    static $cached = null;
    if ($cached !== null) return $cached;

    $cacheFile = __DIR__ . '/.bot_username_cache.txt';
    if (file_exists($cacheFile)) {
        $u = trim((string)@file_get_contents($cacheFile));
        if ($u !== '') {
            $cached = $u;
            return $cached;
        }
    }

    $resp = tg('getMe', []);
    $username = $resp['result']['username'] ?? null;
    if (is_string($username) && $username !== '') {
        @file_put_contents($cacheFile, $username);
        $cached = $username;
        return $cached;
    }

    return null;
}

function getReferralLinkByUserId(int $user_id) {
    $u = getBotUsernameCached();
    if (!$u) return null;
    return "https://t.me/" . $u . "?start=ref_" . $user_id;
}

function recordReferral(int $referrer_id, int $referred_id) {
    if ($referrer_id <= 0 || $referred_id <= 0) return;
    if ($referrer_id === $referred_id) return;

    $pdo = db();

    // فقط یکبار برای هر invited user
    $stmt = $pdo->prepare("INSERT IGNORE INTO referrals (referrer_user_id, referred_user_id, created_at) VALUES (?, ?, ?)");
    $stmt->execute([$referrer_id, $referred_id, time()]);
}

function getReferralStats(int $referrer_id) {
    $pdo = db();
    // بعضی دیتابیس‌ها هنوز ستون final_price_toman را ندارند.
    // برای جلوگیری از Crash، ابتدا با final_price_toman تلاش می‌کنیم و اگر نبود، به price_toman برمی‌گردیم.
    try {
        $sql = "SELECT 
                    COUNT(DISTINCT r.referred_user_id) AS referrals,
                    COUNT(o.id) AS purchases,
                    COALESCE(SUM(CASE WHEN o.final_price_toman > 0 THEN o.final_price_toman ELSE o.price_toman END), 0) AS total_purchase
                FROM referrals r
                LEFT JOIN orders o 
                    ON o.user_id = r.referred_user_id
                   AND o.status = 'completed'
                WHERE r.referrer_user_id = ?";
        $st = $pdo->prepare($sql);
        $st->execute([$referrer_id]);
        $row = $st->fetch(PDO::FETCH_ASSOC) ?: ['referrals'=>0,'purchases'=>0,'total_purchase'=>0];
    } catch (Throwable $e) {
        $sql = "SELECT 
                    COUNT(DISTINCT r.referred_user_id) AS referrals,
                    COUNT(o.id) AS purchases,
                    COALESCE(SUM(o.price_toman), 0) AS total_purchase
                FROM referrals r
                LEFT JOIN orders o 
                    ON o.user_id = r.referred_user_id
                   AND o.status = 'completed'
                WHERE r.referrer_user_id = ?";
        $st = $pdo->prepare($sql);
        $st->execute([$referrer_id]);
        $row = $st->fetch(PDO::FETCH_ASSOC) ?: ['referrals'=>0,'purchases'=>0,'total_purchase'=>0];
    }

    return [
        'referrals' => (int)($row['referrals'] ?? 0),
        'purchases' => (int)($row['purchases'] ?? 0),
        'total_purchase' => (int)($row['total_purchase'] ?? 0),
    ];
}

function hasClaimedReferralGift(int $user_id) {
    $pdo = db();
    $st = $pdo->prepare("SELECT 1 FROM referral_gift_claims WHERE user_id=? LIMIT 1");
    $st->execute([$user_id]);
    return (bool)$st->fetchColumn();
}

function claimReferralGift(int $user_id) {
    $stats = getReferralStats($user_id);
    if ($stats['referrals'] < 1) {
        return ['ok' => false, 'reason' => 'no_referrals'];
    }
    if (hasClaimedReferralGift($user_id)) {
        return ['ok' => false, 'reason' => 'already_claimed'];
    }

    $gift = (int)REFERRAL_MEMBERSHIP_GIFT_TOMAN;
    if ($gift < 1) {
        return ['ok' => false, 'reason' => 'disabled'];
    }

    $cred = walletCredit($user_id, $gift, null, null, ['kind' => 'referral_gift']);
    if (empty($cred['ok'])) {
        return ['ok' => false, 'reason' => 'wallet_failed'];
    }

    $pdo = db();
    $pdo->prepare("INSERT IGNORE INTO referral_gift_claims (user_id, gift_amount_toman, claimed_at) VALUES (?, ?, ?)")
        ->execute([$user_id, $gift, time()]);

    return ['ok' => true, 'gift' => $gift, 'balance' => (int)($cred['balance'] ?? 0)];
}

/**
 * پرداخت پورسانت خرید زیرمجموعه برای یک سفارش خرید (orders)
 */
function processReferralCommissionForOrder(int $order_id) {
    $pdo = db();

    // اگر قبلاً پورسانت داده شده، تکرار نکن
    $chk = $pdo->prepare("SELECT 1 FROM referral_commissions WHERE order_id=? LIMIT 1");
    $chk->execute([$order_id]);
    if ($chk->fetchColumn()) return ['ok' => true, 'skipped' => true];

    // بعضی دیتابیس‌ها هنوز ستون final_price_toman را ندارند.
    // برای جلوگیری از Crash، ابتدا با final_price_toman تلاش می‌کنیم و اگر نبود، به price_toman برمی‌گردیم.
    try {
        $st = $pdo->prepare("SELECT id, user_id, price_toman, final_price_toman, status FROM orders WHERE id=? LIMIT 1");
        $st->execute([$order_id]);
        $o = $st->fetch(PDO::FETCH_ASSOC);
    } catch (Throwable $e) {
        $st = $pdo->prepare("SELECT id, user_id, price_toman, status FROM orders WHERE id=? LIMIT 1");
        $st->execute([$order_id]);
        $o = $st->fetch(PDO::FETCH_ASSOC);
        // سازگاری: اگر final_price_toman نبود، همان price_toman را به‌عنوان مبلغ نهایی در نظر می‌گیریم.
        if (is_array($o) && !isset($o['final_price_toman'])) {
            $o['final_price_toman'] = (int)($o['price_toman'] ?? 0);
        }
    }
    if (!$o) return ['ok' => false, 'reason' => 'order_not_found'];
    if (($o['status'] ?? '') !== 'completed') return ['ok' => false, 'reason' => 'not_completed'];

    $buyer_id = (int)$o['user_id'];
    $ref = $pdo->prepare("SELECT referrer_user_id FROM referrals WHERE referred_user_id=? LIMIT 1");
    $ref->execute([$buyer_id]);
    $referrer_id = (int)$ref->fetchColumn();
    if ($referrer_id < 1) return ['ok' => true, 'skipped' => true];

    $amountBase = (int)$o['final_price_toman'];
    if ($amountBase < 1) $amountBase = (int)$o['price_toman'];
    if ($amountBase < 1) return ['ok' => true, 'skipped' => true];

    $commission = (int)floor($amountBase * ((int)REFERRAL_COMMISSION_PERCENT) / 100);
    if ($commission < 1) return ['ok' => true, 'skipped' => true];

    $cred = walletCredit($referrer_id, $commission, $order_id, null, ['kind' => 'referral_commission', 'order_id' => $order_id, 'referred_user_id' => $buyer_id, 'percent' => (int)REFERRAL_COMMISSION_PERCENT]);
    if (empty($cred['ok'])) return ['ok' => false, 'reason' => 'wallet_failed'];

    $pdo->prepare("INSERT INTO referral_commissions (order_id, referrer_user_id, referred_user_id, amount_toman, created_at) VALUES (?, ?, ?, ?, ?)")
        ->execute([$order_id, $referrer_id, $buyer_id, $commission, time()]);

    return ['ok' => true, 'referrer_id' => $referrer_id, 'commission' => $commission, 'balance' => (int)($cred['balance'] ?? 0)];
}
