<?php
/**
 * Cron Worker - DNet VPN Bot
 *
 * این فایل باید از طریق CronJob سی‌پنل (مثلاً هر 1 دقیقه) اجرا شود.
 * وظایف:
 * 1) مدیریت تایم‌اوت‌های خرید (15 دقیقه)
 * 2) ارسال پیام‌های همگانی (Broadcast) به صورت batch با مدیریت ریت‌لیمیت
 */

require __DIR__ . '/functions.php';

// جلوگیری از اجرای همزمان کران (برای جلوگیری از Too many connections)
$__lockFp = @fopen(__DIR__ . '/.cron.lock', 'c');
if ($__lockFp) {
    if (!@flock($__lockFp, LOCK_EX | LOCK_NB)) {
        // کران قبلی هنوز در حال اجراست
        exit;
    }
    register_shutdown_function(function() use ($__lockFp) {
        @flock($__lockFp, LOCK_UN);
        @fclose($__lockFp);
    });
}

ignore_user_abort(true);
@set_time_limit(0);

$pdo = db();
$now = time();

// --------------------------------------------------
// 1) مدیریت تایم‌اوت‌ها
// --------------------------------------------------
try {
    $limit = 200;
    $stmt = $pdo->prepare("SELECT chat_id, state, expires_at FROM states WHERE expires_at IS NOT NULL AND expires_at > 0 AND expires_at < ? ORDER BY expires_at ASC LIMIT {$limit}");
    $stmt->execute([$now]);
    $expired = $stmt->fetchAll();

    foreach ($expired as $row) {
        $cid = (int)$row['chat_id'];
        $st = (string)$row['state'];

        // تایم‌اوت خرید
        if ($st === 'buy_wait_receipt') {
            sendMessage($cid, "⏱ مهلت پرداخت شما به پایان رسید و خرید به صورت خودکار لغو شد.", startKeyboardFor($cid));
        }

        // تایم‌اوت شارژ کیف پول
        if ($st === 'wallet_topup_wait_receipt') {
            sendMessage($cid, "⏱ مهلت شارژ کیف پول شما به پایان رسید و درخواست به صورت خودکار لغو شد.", startKeyboardFor($cid));
        }

        clearState($cid);
        usleep(20000);
    }
} catch (Throwable $e) {
    log_error('CRON TIMEOUTS ERROR', ['msg' => $e->getMessage()]);
}

// --------------------------------------------------
// 2) ارسال همگانی (Broadcast)
// --------------------------------------------------
try {
    $bc = $pdo->query("SELECT * FROM broadcasts WHERE status IN ('pending','sending') ORDER BY id ASC LIMIT 1")->fetch();
    if ($bc) {
        $bid = (int)$bc['id'];

        if ($bc['status'] === 'pending') {
            $total = (int)$pdo->query("SELECT COUNT(*) FROM users WHERE is_blocked=0")->fetchColumn();
            $stmt = $pdo->prepare("UPDATE broadcasts SET status='sending', total_targets=?, started_at=?, cursor_user_id=0 WHERE id=?");
            $stmt->execute([$total, $now, $bid]);
            $bc['status'] = 'sending';
            $bc['total_targets'] = $total;
            $bc['cursor_user_id'] = 0;
            $bc['started_at'] = $now;
        }

        $batch = 120;           // تعداد کاربران در هر اجرای کران
        $sleepUs = 120000;      // تاخیر بین ارسال‌ها (برای کاهش ریت‌لیمیت)

        $cursor = (int)$bc['cursor_user_id'];
        $stmt = $pdo->prepare("SELECT id, chat_id FROM users WHERE is_blocked=0 AND id > ? ORDER BY id ASC LIMIT {$batch}");
        $stmt->execute([$cursor]);
        $targets = $stmt->fetchAll();

        if (!$targets) {
            // پایان ارسال
            $stmt = $pdo->prepare("UPDATE broadcasts SET status='completed', finished_at=? WHERE id=?");
            $stmt->execute([$now, $bid]);

            // گزارش برای ادمین
            $fresh = $pdo->prepare("SELECT * FROM broadcasts WHERE id=? LIMIT 1");
            $fresh->execute([$bid]);
            $b = $fresh->fetch();

            $report = "📢 گزارش ارسال همگانی #{$bid}\n\n";
            $report .= "کل کاربران هدف: <b>" . (int)$b['total_targets'] . "</b>\n";
            $report .= "ارسال موفق: <b>" . (int)$b['sent_count'] . "</b>\n";
            $report .= "ناموفق: <b>" . (int)$b['failed_count'] . "</b>\n";
            $report .= "بلاک‌شده‌ها: <b>" . (int)$b['blocked_count'] . "</b>\n";

            sendMessage(ADMIN_ID, $report, adminMenuKeyboard());
            echo "DONE\n";
            exit;
        }

        $sentAdd = 0;
        $failAdd = 0;
        $blockedAdd = 0;
        $lastId = $cursor;
        $lastError = null;

        foreach ($targets as $t) {
            $uid = (int)$t['id'];
            $cid = (int)$t['chat_id'];
            $lastId = $uid;

            $res = copyMessage($cid, (int)$bc['source_chat_id'], (int)$bc['source_message_id']);

            // مدیریت 429
            if ($res && empty($res['ok']) && (int)($res['error_code'] ?? 0) === 429) {
                $retry = (int)($res['parameters']['retry_after'] ?? 1);
                if ($retry > 0 && $retry < 60) {
                    sleep($retry);
                } else {
                    usleep(500000);
                }
                $res = copyMessage($cid, (int)$bc['source_chat_id'], (int)$bc['source_message_id']);
            }

            if ($res && !empty($res['ok'])) {
                $sentAdd++;
            } else {
                $failAdd++;
                $errCode = (int)($res['error_code'] ?? 0);
                $desc = (string)($res['description'] ?? '');
                $lastError = "{$errCode}: {$desc}";

                // اگر کاربر ربات را بلاک کرده باشد
                if ($errCode === 403 || stripos($desc, 'blocked by the user') !== false || stripos($desc, 'user is deactivated') !== false || stripos($desc, 'chat not found') !== false) {
                    $pdo->prepare("UPDATE users SET is_blocked=1 WHERE id=?")->execute([$uid]);
                    $blockedAdd++;
                }
            }

            // آپدیت cursor برای resume امن
            $pdo->prepare("UPDATE broadcasts SET cursor_user_id=? WHERE id=?")->execute([$lastId, $bid]);

            usleep($sleepUs);
        }

        $upd = $pdo->prepare(
            "UPDATE broadcasts SET sent_count = sent_count + ?, failed_count = failed_count + ?, blocked_count = blocked_count + ?, last_error = ? WHERE id=?"
        );
        $upd->execute([$sentAdd, $failAdd, $blockedAdd, $lastError, $bid]);
    }
} catch (Throwable $e) {
    log_error('CRON BROADCAST ERROR', ['msg' => $e->getMessage()]);
}

// --------------------------------------------------
// 3) شارژ همگانی کیف پول (Wallet Bulk Credits)
// --------------------------------------------------
try {
    // اگر جدول وجود ندارد، سکوت
    $tbl = $pdo->query("SHOW TABLES LIKE 'wallet_bulk_credits'")->fetchColumn();
    if ($tbl) {
        $job = $pdo->query("SELECT * FROM wallet_bulk_credits WHERE status IN ('pending','sending') ORDER BY id ASC LIMIT 1")->fetch();
        if ($job) {
            $jid = (int)$job['id'];

            if ($job['status'] === 'pending') {
                $total = (int)$pdo->query("SELECT COUNT(*) FROM users WHERE is_blocked=0")->fetchColumn();
                $stmt = $pdo->prepare("UPDATE wallet_bulk_credits SET status='sending', total_targets=?, started_at=?, cursor_user_id=0 WHERE id=?");
                $stmt->execute([$total, $now, $jid]);
                $job['status'] = 'sending';
                $job['total_targets'] = $total;
                $job['cursor_user_id'] = 0;
                $job['started_at'] = $now;
            }

            $batch = 80;
            $sleepUs = 140000;

            $cursor = (int)$job['cursor_user_id'];
            $amount = (int)$job['amount_toman'];

            $stmt = $pdo->prepare("SELECT id, chat_id FROM users WHERE is_blocked=0 AND id > ? ORDER BY id ASC LIMIT {$batch}");
            $stmt->execute([$cursor]);
            $targets = $stmt->fetchAll();

            if (!$targets) {
                $pdo->prepare("UPDATE wallet_bulk_credits SET status='completed', finished_at=? WHERE id=?")->execute([$now, $jid]);

                $fresh = $pdo->prepare("SELECT * FROM wallet_bulk_credits WHERE id=? LIMIT 1");
                $fresh->execute([$jid]);
                $j = $fresh->fetch();

                $report = "💰 گزارش شارژ همگانی #{$jid}\n\n";
                $report .= "مبلغ شارژ: <b>" . formatToman((int)($j['amount_toman'] ?? 0)) . " تومان</b>\n";
                $report .= "کل کاربران هدف: <b>" . (int)($j['total_targets'] ?? 0) . "</b>\n";
                $report .= "شارژ موفق: <b>" . (int)($j['credited_count'] ?? 0) . "</b>\n";
                $report .= "پیام ارسال‌شده: <b>" . (int)($j['notified_count'] ?? 0) . "</b>\n";
                $report .= "ناموفق (ارسال پیام): <b>" . (int)($j['failed_count'] ?? 0) . "</b>\n";
                $report .= "بلاک‌شده‌ها: <b>" . (int)($j['blocked_count'] ?? 0) . "</b>";

                $adminChat = (int)($j['admin_chat_id'] ?? ADMIN_ID);
                if ($adminChat < 1) $adminChat = ADMIN_ID;
                sendMessage($adminChat, $report, adminMenuKeyboard());
                echo "DONE_BULK\n";
                exit;
            }

            $creditedAdd = 0;
            $notifiedAdd = 0;
            $failAdd = 0;
            $blockedAdd = 0;
            $lastId = $cursor;
            $lastError = null;

            foreach ($targets as $t) {
                $uid = (int)$t['id'];
                $cid = (int)$t['chat_id'];
                $lastId = $uid;

                $cred = walletCredit($uid, $amount, null, null, ['kind' => 'bulk_credit', 'job_id' => $jid]);
                if (empty($cred['ok'])) {
                    $failAdd++;
                    $lastError = 'wallet_credit_failed';
                    $pdo->prepare("UPDATE wallet_bulk_credits SET cursor_user_id=? WHERE id=?")->execute([$lastId, $jid]);
                    usleep($sleepUs);
                    continue;
                }

                $creditedAdd++;

                $msg = "💰 کیف پول شما از طرف ادمین شارژ شد.\n\n";
                $msg .= "مبلغ: <b>" . formatToman($amount) . " تومان</b>\n";
                $msg .= "موجودی جدید: <b>" . formatToman((int)($cred['balance'] ?? 0)) . " تومان</b>";

                $res = sendMessage($cid, $msg, userMenuKeyboard());

                // مدیریت 429
                if ($res && empty($res['ok']) && (int)($res['error_code'] ?? 0) === 429) {
                    $retry = (int)($res['parameters']['retry_after'] ?? 1);
                    if ($retry > 0 && $retry < 60) {
                        sleep($retry);
                    } else {
                        usleep(500000);
                    }
                    $res = sendMessage($cid, $msg, userMenuKeyboard());
                }

                if ($res && !empty($res['ok'])) {
                    $notifiedAdd++;
                } else {
                    $failAdd++;
                    $errCode = (int)($res['error_code'] ?? 0);
                    $desc = (string)($res['description'] ?? '');
                    $lastError = "{$errCode}: {$desc}";

                    if ($errCode === 403 || stripos($desc, 'blocked by the user') !== false || stripos($desc, 'user is deactivated') !== false || stripos($desc, 'chat not found') !== false) {
                        $pdo->prepare("UPDATE users SET is_blocked=1 WHERE id=?")->execute([$uid]);
                        $blockedAdd++;
                    }
                }

                $pdo->prepare("UPDATE wallet_bulk_credits SET cursor_user_id=? WHERE id=?")->execute([$lastId, $jid]);

                usleep($sleepUs);
            }

            $upd = $pdo->prepare(
                "UPDATE wallet_bulk_credits SET credited_count = credited_count + ?, notified_count = notified_count + ?, failed_count = failed_count + ?, blocked_count = blocked_count + ?, last_error = ? WHERE id=?"
            );
            $upd->execute([$creditedAdd, $notifiedAdd, $failAdd, $blockedAdd, $lastError, $jid]);
        }
    }
} catch (Throwable $e) {
    log_error('CRON WALLET BULK ERROR', ['msg' => $e->getMessage()]);
}

// --------------------------------------------------
// 4) بروزرسانی مصرف/انقضای کانفیگ‌ها + هشدار 10% باقی‌مانده (زمان یا حجم)
// --------------------------------------------------
try {
    // این کران معمولاً هر 1 دقیقه اجرا می‌شود؛ ما هر 10 دقیقه یکبار رفرش می‌کنیم.
    $stampFile = __DIR__ . '/._usage_refresh_last';
    $lastRun = 0;
    if (is_file($stampFile)) {
        $lastRun = (int)@file_get_contents($stampFile);
    }

    if (($now - $lastRun) >= 600) {
        @file_put_contents($stampFile, (string)$now);
        ensureConfigUsageCacheTable();

        // فقط کانفیگ‌های فعال
        $stmt = $pdo->query(
            "SELECT c.id, c.user_id, c.name, c.link, c.data_gb, c.duration_days, c.created_at, u.chat_id\n"
            . "FROM configs c JOIN users u ON u.id=c.user_id\n"
            . "WHERE c.active=1 ORDER BY c.id ASC"
        );
        $cfgs = $stmt->fetchAll();

        foreach ($cfgs as $c) {
            $cfgId = (int)$c['id'];
            $chatId = (int)$c['chat_id'];
            $name = (string)$c['name'];
            $link = (string)$c['link'];

            $usage = fetchConfigUsage($link);
            if (!empty($usage['ok'])) {
                upsertConfigUsageCache($cfgId, $usage);

                // برای هشدارها، کش فعلی را بخوانیم
                $row = $pdo->prepare("SELECT * FROM config_usage_cache WHERE config_id=? LIMIT 1");
                $row->execute([$cfgId]);
                $cache = $row->fetch();

                if ($cache) {
                    $total = (int)$cache['total_bytes'];
                    $used = (int)$cache['used_bytes'];
                    $percent = (int)$cache['percent_used'];
                    $warn80 = (int)$cache['warn80_sent'];
                    $warn10 = (int)$cache['warn10_sent'];
                    $expiredSent = (int)($cache['expired_sent'] ?? 0);
                    $daysLeft = array_key_exists('days_left', $cache) ? (is_null($cache['days_left']) ? null : (int)$cache['days_left']) : null;
                    $expireTs = array_key_exists('expire_ts', $cache) ? (is_null($cache['expire_ts']) ? null : (int)$cache['expire_ts']) : null;
                    $durationDays = (int)($c['duration_days'] ?? 0);

                    $createdAt = (int)($c['created_at'] ?? 0);
                    // اگر پنل expire_ts نداد، از تاریخ خرید + مدت پلن محاسبه می‌کنیم
                    if (($expireTs === null || $expireTs <= 0) && $createdAt > 0 && $durationDays > 0) {
                        $expireTs = $createdAt + ($durationDays * 86400);
                        $daysLeft = (int)ceil(($expireTs - $now) / 86400);
                    }



                    // اگر حجم نامحدود باشد (total=0) هشدار حجمی نمی‌دهیم
                    // ✅ بررسی انقضا (زمان یا حجم) - برای همه سرویس‌ها
                    $expiredByTime = (($expireTs !== null && $expireTs > 0 && $expireTs <= $now) || ($daysLeft !== null && $daysLeft <= 0));
                    $expiredByVolume = ($total >= 1024 * 1024 && $used >= $total);

                    if (($expiredByTime || $expiredByVolume) && $expiredSent === 0) {
                        $txt = "⛔️ <b>کانفیگ شما منقضی شد</b>

";
                        $txt .= "کانفیگ <b>" . htmlspecialchars($name) . "</b> منقضی شد.
";
                        $txt .= "در صورت تمایل برای <b>تمدید</b> اقدام نمایید.";

                        $kb = ['inline_keyboard' => [[['text' => '♻️ تمدید همین سرویس', 'callback_data' => 'renew:select:' . $cfgId]]]];
                        sendMessage($chatId, $txt, $kb);

                        $pdo->prepare("UPDATE config_usage_cache SET expired_sent=1, last_warned_at=? WHERE config_id=?")
                            ->execute([$now, $cfgId]);
                        usleep(120000);
                    }

                    
                    // ✅ هشدار 10% باقی‌مانده از نظر زمان (بر اساس expire_ts واقعی، برای همه لینک‌ها)
                    if ($expiredSent === 0 && $warn10 === 0 && $expireTs !== null && $expireTs > $now) {
                        $remainSec = (int)($expireTs - $now);
                        $remainDays = (int)ceil($remainSec / 86400);

                        $shouldWarn = false;
                        // اگر created_at معتبر باشد، درصد زمان باقی‌مانده را دقیق حساب می‌کنیم
                        if ($createdAt > 0 && $expireTs > $createdAt) {
                            $totalSec = (int)($expireTs - $createdAt);
                            if ($totalSec > 0) {
                                $remainRatio = $remainSec / $totalSec;
                                if ($remainRatio <= 0.10) $shouldWarn = true;
                            }
                        } elseif ($durationDays > 0 && $daysLeft !== null) {
                            // فالبک: بر اساس مدت پلن (اگر expire_ts از پنل نیامده باشد یا created_at نداریم)
                            $threshold = (int)max(1, ceil($durationDays * 0.1));
                            if ($daysLeft <= $threshold) $shouldWarn = true;
                            $remainDays = (int)$daysLeft;
                        }

                        if ($shouldWarn) {
                            $txt = "⚠️ <b>هشدار</b>

";
                            $txt .= "فقط <b>10%</b> از بسته کانفیگ <b>" . htmlspecialchars($name) . "</b> باقی مانده است.
";
                            if ($remainDays >= 0) {
                                $txt .= "⏳ زمان باقی‌مانده: <b>" . (int)$remainDays . " روز</b>

";
                            }
                            $txt .= "برای ادامه استفاده، تمدید را انجام دهید.";

                            $kb = ['inline_keyboard' => [[['text' => '♻️ تمدید همین سرویس', 'callback_data' => 'renew:select:' . $cfgId]]]];
                            sendMessage($chatId, $txt, $kb);

                            $pdo->prepare("UPDATE config_usage_cache SET warn10_sent=1, last_warned_at=? WHERE config_id=?")
                                ->execute([$now, $cfgId]);
                            usleep(120000);
                        }
                    }

                    if ($expiredByTime || $expiredByVolume) {
                        continue;
                    }

                    if ($total >= 1024 * 1024) {
                        $remainPercent = 100 - $percent;
                        // کمتر از 10% باقی‌مانده
                        if ($remainPercent <= 10 && $warn10 === 0) {
                            $txt = "⚠️ <b>هشدار</b>

";
                            $txt .= "فقط <b>10%</b> از بسته کانفیگ <b>" . htmlspecialchars($name) . "</b> باقی مانده است.
";
                            $txt .= "مصرف: <b>" . htmlspecialchars(formatGb($used)) . "</b> از <b>" . htmlspecialchars(formatGb($total)) . "</b> ({$percent}%)\n\n";
                            $txt .= "برای ادامه استفاده، تمدید را انجام دهید.";
                            $kb = ['inline_keyboard' => [[['text' => '🔄 تمدید همین سرویس', 'callback_data' => 'renew:select:' . $cfgId]]]];
                            sendMessage($chatId, $txt, $kb);

                            $pdo->prepare("UPDATE config_usage_cache SET warn10_sent=1, last_warned_at=? WHERE config_id=?")
                                ->execute([$now, $cfgId]);
                            usleep(120000);
                        }
                    }
                }
            }

            // ریت‌لیمیت سبک
            usleep(80000);
        }
    }

// --------------------------------------------------
// 4-b) یادآوری عدم اتصال: اگر ۱ ساعت از ارسال کانفیگ گذشته و مصرف صفر است
// --------------------------------------------------
try {
    ensureConfigUsageCacheTable();

    $q = $pdo->prepare(
        "SELECT cuc.config_id, c.name, u.chat_id\n"
        . "FROM config_usage_cache cuc\n"
        . "JOIN configs c ON c.id=cuc.config_id\n"
        . "JOIN users u ON u.id=c.user_id\n"
        . "WHERE c.active=1\n"
        . "  AND cuc.delivered_at IS NOT NULL AND cuc.delivered_at > 0\n"
        . "  AND cuc.no_use_notice_sent=0\n"
        . "  AND cuc.used_bytes=0\n"
        . "  AND cuc.last_checked > ?\n"
        . "  AND cuc.delivered_at < ?\n"
        . "ORDER BY cuc.delivered_at ASC\n"
        . "LIMIT 60"
    );
    // last_checked حداقل در ۲ ساعت اخیر باشد تا خطای false-positive ندهیم
    $q->execute([$now - 7200, $now - 3600]);
    $rows = $q->fetchAll();

    foreach ($rows as $r) {
        $cid = (int)$r['chat_id'];
        $cfgId = (int)$r['config_id'];
        $name = (string)$r['name'];

        $txt = "💛 کاربر عزیز\n\n";
        $txt .= "⏱ ۱ ساعت از دریافت کانفیگ «<b>" . htmlspecialchars($name) . "</b>» گذشته و هنوز به کانفیگ متصل نشده‌اید.\n\n";
        $txt .= "اگر مشکلی در روند اتصال دارید به قسمت «📡 نحوه اتصال» مراجعه نمایید.\n";
        $txt .= "در صورت حل نشدن مشکل، به «🆘 پشتیبانی» پیام دهید.\n\n";
        $txt .= "با تشکر 💛";

        sendMessage($cid, $txt, userMenuKeyboard());

        $pdo->prepare("UPDATE config_usage_cache SET no_use_notice_sent=1 WHERE config_id=?")->execute([$cfgId]);
        usleep(150000);
    }
} catch (Throwable $e2) {
    log_error('CRON NO-USE REMINDER ERROR', ['msg' => $e2->getMessage()]);
}
} catch (Throwable $e) {
    log_error('CRON USAGE REFRESH ERROR', ['msg' => $e->getMessage()]);
}

echo "OK\n";
